; Test for 18F442 (but should run on any 18FXX2)
; RC Osc
; connect led/resistor to ground on RB0
; Malcolm Wiles, 10 Dec 04

; This is possibly the world's most complicated flash led program.
; Its main purpose is to illustrate some new features of the 18F family.

; First, 64 bytes of data memory are initialised to 0x10, 0x11.. 0x4F
; using FSR0 in post increment mode.  The DATABUF address is deliberately
; chosen to wrap across a data memory bank boundary, to show that this
; doesn't matter with indirect addressing.

; Next it starts the led flashing very fast controlled by TMR0 in 8-bit
; mode, the flashing driven by T0IF interrupts.  If you actually run it,
; you will only see the led appear to be on dimly, the flash rate is too
; fast to see with the eye.
; The timer interrupts are intended to cause general annoyance to the rest of
; the code, and highlight any weaknesses in the program (or chip design)! 

; The next short sequence shows how to implement a table read.

; 64 bytes of program memory are erased (CODEBUF), and the data written to
; DATABUF are copied to into CODEBUF in 8 writes of 8 bytes each.  FSR1 is
; used as the DATABUF pointer, and tblwt with post increment is used to 
; write program memory.  Observe that the pointer has
; to be adjusted to point back into the correct block in a couple of
; places using tblrd. "critical section" macros are used to prevent the
; write unlock sequences from being disrupted by the TMR0 interrupts.

; DATABUF is then copied to EEPROM data memory. (This was originally done so
; that data EEPROM could be read out via a programmer as a check that everything
; was working OK.) FSR2 is the DATABUF pointer.  After the first byte write has
; been initiated, the code runs on into the main loop, and the remaining bytes
; are copied by the isr following each EEIF interrupt.

; The isr does not save context, uses fast return, and corrupts W.  The main loop
; adds W=1 to each of two counters, and checks that they remain consistent. If
; automatic context save does not work the counters would get out of sync and the
; main loop would disable all interrupts, stopping the led flashing (this doesn't
; happen!).

; Finally when the EEPROM write is completed, TMR0 is put into 16-bit mode
; "on-the-fly", which slows the led flash rate down to 1Hz or so.


        LIST    p=18f442,r=dec

	include P18F442.inc
 
B	equ	1
F 	equ	1

; macros

enter_critical_section	macro
	bcf	INTCON,GIE,A
	endm

leave_critical_section	macro
	bsf	INTCON,GIE,A
	endm

	__config H'300000', H'00'		; config
	__config H'300001', H'23'		; config
	__config H'300002', H'08'		; config
	__config H'300003', H'00'		; config
	__config H'300004', H'00'		; config
	__config H'300005', H'00'		; config
	__config H'300006', H'80'		; config
	__config H'300007', H'00'		; config
	__config H'300008', H'0F'		; config
	__config H'300009', H'C0'		; config
	__config H'30000A', H'0F'		; config
	__config H'30000B', H'E0'		; config
	__config H'30000C', H'0F'		; config
	__config H'30000D', H'40'		; config


 ;  Vars (bank 0)

TEMP1	equ	0x0
TEMP2	equ	0x1
TEMP3	equ	0x2
TEMP4	equ	0x3
EECT	equ	0x4		; EEPROM bytes to write
LOOP	equ	0x5		; for LCD table read

; buffer addresses

DATABUF equ	0x01E0		; data memory address
CODEBUF	equ	0x0180		; must be 64 byte aligned

; code

	org	0		; reset vector
	goto	init
	
	org	0008h		; hi interrupt vector
	bra	isr
	org	0018h
	retfie			; lo interrupt vector

; initialise PIC

	org	0020h	
init:	
	movlb	15		; set bank 15 (illustrate banked addressing)
	movlw	0xFF
	movwf	LATB,B		; set latches
	clrf	TRISB,B		; all PORTB outputs
	movlb	1		; set bank 1

; Timer 0 setup

	movlw	0xF8		; clear prescale bits
	andwf	T0CON,F,A		
	movlw	1		; prescale /4
;	iorwf	T0CON,F,A
	bcf	T0CON,PSA,A	; assign prescaler
	bsf	T0CON,T08BIT,A	; 8 bit mode
	bcf	T0CON,T0CS,A	; timer mode
	clrf	TMR0H,A		; clear counter hi byte latch
	clrf	TMR0L,A		; clear counter	
	bcf	INTCON, T0IF,A	; ensure flag bit clear

; set porta inputs

	movlw	6
	movwf	ADCON1,A
	
; data initialisation

	clrf	TEMP1,A
	clrf	TEMP2,A

; initialise data memory buffer DATABUF - 64 bytes long

	movlw	64		; bytes to set
	movwf	TEMP3,A
	lfsr	FSR0,DATABUF	; address to start
	movlw	0x10		; starting data value
B1:
	movwf	POSTINC0,A	; write to FSR0 and then bump FSR0
	incf	WREG,F,A	; bump value written
	decfsz	TEMP3,F,A	; count bytes to write
	bra	B1

; start interrupts

	bcf	RCON,IPEN,A	; no priority interrupts
	bsf	INTCON,T0IE,A	; enable TMR0 interrupt
	bsf	INTCON,PEIE,A	; enable periph interrupts
	bsf	INTCON,GIE,A	; enable global interrupts

; How to do a table read properly
; This code will work from any address with a data table located at any address

	movlw	UPPER TABLCD	; load upper, high and low parts of
	movwf	TBLPTRU,A	; the TABLCD address into TBLPTR
	movlw	HIGH TABLCD
	movwf	TBLPTRH,A
	movlw	LOW TABLCD
	movwf	TBLPTRL,A
	movlw	8		; read 8 bytes
	movwf	LOOP,A		; initialise reads counter
RDTAB:
	tblrd*+			; read table, post increment address
	movf	TABLAT,W,A	; get table data from latch to W
	rcall	LCDOUT		; initialise LCD
	decfsz	LOOP,F,A	; done yet?
	bra	RDTAB		; no

; copy data set up in DATABUF to program memory buffer CODEBUF

	lfsr	FSR1,DATABUF	; data mem pointer
	movlw	UPPER CODEBUF
	movwf	TBLPTRU,A	; prog mem addr
	movlw	HIGH CODEBUF
	movwf	TBLPTRH,A
	movlw	LOW CODEBUF
	movwf	TBLPTRL,A

; first erase 64 bytes of CODEBUF

	bsf	EECON1,EEPGD,A	; point to flash
	bcf	EECON1,CFGS,A	; point to flash
	bsf	EECON1,WREN,A	; write enable
	bsf	EECON1,FREE,A	; erase

	enter_critical_section
	movlw	0x55		; unlock sequence
	movwf	EECON2,A
	movlw	0xAA
	movwf	EECON2,A
	bsf	EECON1,WR,A	; initiate erase
	leave_critical_section

	bcf	EECON1,WREN,A	; for safety disable write

; now set up the copy

	movlw	8
	movwf	TEMP3,A		; 8 groups
B2:
	movlw	8
	movwf	TEMP4,A		; 8 bytes per group
B3:
	movf	POSTINC1,W,A	; load data byte
	movwf	TABLAT,A	; into program write latch
	tblwt*+			; and write it post inc
	decfsz	TEMP4,F,A
	bra	B3
	tblrd*-			; put table ptr back into first 8 byte block,
				; anywhere will do (ignore data byte read)
	bsf	EECON1,EEPGD,A	; point to flash
	bcf	EECON1,CFGS,A	; point to flash
	bsf	EECON1,WREN,A	; write enable
	bcf	EECON1,FREE,A	; write not erase

	enter_critical_section
	movlw	0x55		; unlock sequence
	movwf	EECON2,A
	movlw	0xAA
	movwf	EECON2,A
	bsf	EECON1,WR,A	; initiate write
	leave_critical_section

	tblrd*+			; put table pointer back again
	decfsz	TEMP3,F,A	; count groups done
	bra	B2
	bcf	EECON1,WREN,A	; safety disable write

; copy DATABUF to data EEPROM

	movlw	0
	movwf	EEADR,A		; offset 0 in EEPROM
	movlw	64		
	movwf	EECT,A		; bytes to move
	lfsr	FSR2, DATABUF	; pointer to databuf
	bcf	EECON1,EEPGD,A	; point to EEPROM
	bcf	EECON1,CFGS,A	; point to flash or EEPROM
	bsf	EECON1,WREN,A	; write enable
	bcf	EECON1,FREE,A	; write not erase
B10:
	bcf	PIR2,EEIF,A
	bsf	PIE2,EEIE,A	; enable EEPROM write int

	movf	POSTINC2,W,A	; load first data byte
	movwf	EEDATA,A

	enter_critical_section		
	movlw	0x55		; unlock sequence
	movwf	EECON2,A
	movlw	0xAA
	movwf	EECON2,A
	bsf	EECON1,WR,A	; initiate first write
	leave_critical_section
				; remaining 63 bytes copied by the isr

; main loop

main:
	btfss	PIE2,EEIE,A	; if isr has finished writing eeprom..
	bcf	T0CON,T08BIT,A	; ..set TMR0 to 16 bit mode
	movlw	1		; add 1 to temp1
	addwf	TEMP1,F,A
	addwf	TEMP2,F,A	; and to temp2
	movf	TEMP1,W,A
	xorwf	TEMP2,W,A	; temp1 should = temp2
	bz	main		; Z is set if they are equal
	bcf	INTCON,GIE,A	; if not, stop led flashing
	GOTO	main

; interrupt service routine (isr)

isr:
	movlw	3		; deliberately corrupt W

; timer interrupt

	btfss	INTCON, T0IF, A	; test if TMR0 int
	bra	isr2		; not a TMR0 int
	
	bcf	INTCON, T0IF, A	; clear the interrupt
	btg	LATB,0,A	; toggle bit to flash led on/off

; eeprom write complete

isr2:	btfss	PIR2,EEIF,A	; test if EEPROM write complete
	bra	isr3

	bcf	PIR2,EEIF,A	; clear the interrupt
	dcfsnz	EECT,F,A	; bytes left to write
	bra	isr2a		; done if zero
	incf	EEADR,F,A	; bump address
	movf	POSTINC2,W,A	; load next data byte
	movwf	EEDATA,A		
	movlw	0x55		; unlock sequence
	movwf	EECON2,A	; this is the isr so
	movlw	0xAA		; no need for a critical section
	movwf	EECON2,A
	bsf	EECON1,WR,A	; initiate next write
	bra	isr3

isr2a:				; all data bytes written
	bcf	EECON1,WREN,A	; safety disable write
	bcf	PIE2,EEIE,A	; done so disable EEIF int

isr3:
exit:
	retfie	FAST		; exit isr with automatic context reload

; Just a dummy routine for demo purposes

LCDOUT	return

; LCD initialisation table - copied from John Becker's tk3PIC18Fdemo.asm,
; EPE March 05.
; Note even no of definitions per line to avoid padding by the assembler

TABLCD	db B'00110011', B'00110011'
	db B'00110010', B'00101100'
	db B'00000110', B'00001100'
	db B'00000001', B'00000010'
 
	END
