/*
 * File:    PIC24F-DigitalPot.c
 *
 *          This is a driver for the Microchip Digital Potentiometer MCP4151
 *          A Single Volatile 8-bit Potentiometer
 *
 * Author:  Mike O'Keeffe
 * 
 * Created on 30th July 2018
 * 
 */

#include "mcc_generated_files/mcc.h"
#include "PIC24F-DigitalPot.h"
#define FCY 16000000UL       // instruction cycle Hrtz
#include <libpic30.h>

#define POT_VOL_WIPER0 0x00
#define POT_VOL_TCON 0x40
#define POT_VOL_STAT 0x50       // not really used on the 4151
#define POT_MAX_WIPE 0xFF       // 100K ohms (technically 0x100h is 100K ohms, 0xFF is 99,610 ohms)
#define POT_MIN_WIPE 0x00       // 0 ohms
#define POT_MID_WIPE 0x80       // 50K ohms
#define POTRD_DATA 0x11
#define POTWR_DATA 0x00
#define POT_INCR 0x01
#define POT_DECR 0x10


/*
 * SPI interface
 * Mode 0,0
 *  SCK idle state = low, data clocked on SDI on rising edge of SCK
 *      clocked out on SDO on falling edge of SCK
 * 
 * Mode 1,1
 *  SCK idle state = high, data clocked on SDI on rising edge of SCK
 *      clocked out on SDO on falling edge of SCK
 * 
 */

void digPotInit(void) {
    
    DISP_CSN_SetHigh();
    DIG_POT_CSN_SetLow();
    __delay_ms(100);
    
    // Write 0x0F to TCON[3:0] = R0HW, R0A, R0W, R0B
    writePot_Data(POT_VOL_TCON, 0x0F);  // Not needed
    
    // Set default value to digital potentiometer
    writePot_Value(POT_MID_WIPE);     // for 100K pot, this should be 50K
    
    __delay_ms(5);
    DIG_POT_CSN_SetHigh(); 
}

void writePot_Value(uint8_t data) { 
    /*
     * Write Operation 2 bytes, command and data
     * 
     * |A3|A2|A1|A0|C0|C1|D9|   |D8|D7|D6|D5|D4|D3|D2|D1|D0|
     * 
     * A3:A0 are address to be written to
     * C0:C1 is the command (write, read, increment and decrement)
     * D9:D0 is the 9 bit data byte (D9 is not used, D8 allows full range to 0x100h, which is not needed)
     * 
     */

    uint8_t write_op = 0x00;    // For full range, change to 0x01 and data = 0x00

    DISP_CSN_SetHigh();
    DIG_POT_CSN_SetLow();
    MSSP1_SPI_Exchange8bit(write_op);
    MSSP1_SPI_Exchange8bit(data);
    __delay_ms(5);
    DIG_POT_CSN_SetHigh();
}

void writePot_Data(uint8_t command, uint8_t data) {
    
    DISP_CSN_SetHigh();
    DIG_POT_CSN_SetLow();
    MSSP1_SPI_Exchange8bit(command);
    MSSP1_SPI_Exchange8bit(data);
    __delay_ms(5);
    DIG_POT_CSN_SetHigh();
}