/**
 *******************************************************************
 * 
 * PIC Sinewave Generator
 * 
 * Author: Mike P. O'Keeffe
 * 
 * Everyday Practical Electronics - PIC 'n Mix
 * 
 * Description:
 * 
 * PIC: 16F1829
 * Compiler: XC8 v1.4
 * IDE: MPLABX v3.65
 *
 * Board: PIC Sinewave Generator
 * Date: 01.08.2017
 *
 * Notes: 
 *  Based on Original design by Roman Black
 *      www.romanblack.com/onesec/Sine1kHz.htm
 * 
 *  PIC pins;
 *  RA2 in = switch, using internal RB0 pullup resistor
 *  RC0 = SINE WAVE MODE
 *  RC1 = SQUARE WAVE MODE
 *  RC2 = TRIANGLE WAVE MODE
 *  RC3 = CALIBRATION MODE (Removed)
 *  RC5 = PWM OUTPUT
 * 
 * ******************************************************************
 */

#include <htc.h>                                     //PIC hardware mapping
#define _XTAL_FREQ 500000                            //Used by the XC8 delay_ms(x) macro

#define DOWN                0
#define UP                  1
#define SWITCH              PORTAbits.RA2
#define PULL_UPS                                    //if this is uncommented, the trace under JP5 can be cut
                                                    //with no affect on the output
#define SINE_WAVE           0
#define SQUARE_WAVE         1
#define TRI_WAVE            2
#define CAL_WAVE            3

//config bits that are part-specific for the PIC16F1829
__CONFIG(FOSC_INTOSC & WDTE_OFF & PWRTE_OFF & MCLRE_OFF & CP_OFF & CPD_OFF & BOREN_ON & CLKOUTEN_OFF & IESO_OFF & FCMEN_OFF);
__CONFIG(WRT_OFF & PLLEN_OFF & STVREN_OFF & LVP_OFF);


unsigned char pwmstep;      // the PWM step 0-49
unsigned char pwmval;       // the PWM value to be used for this step
unsigned char debounce;     // used for debouncing switch input
unsigned int pwmMode = 0;

const unsigned char sine[50] = {56,62,68,74,79,84,88,92,95,97,
                                99,99,99,99,97,95,92,88,84,79,
                                74,68,62,56,49,43,37,31,25,20,
                                15,11,7,4,2,0,0,0,0,2,
                                4,7,11,15,20,25,31,37,43,50};

const unsigned char square[50] = {0,0,0,0,0,0,0,0,0,0,
                                0,0,0,0,0,0,0,0,0,0,
                                0,0,0,0,0,
                                100,100,100,100,100,100,100,100,100,100,
                                100,100,100,100,100,100,100,100,100,100,
                                100,100,100,100,100};

const unsigned char triangle[50] = {0,4,8,12,16,20,24,28,32,36,
                                40,44,48,52,56,60,64,68,72,76,
                                80,84,88,92,96,100,96,92,88,84,
                                80,76,72,68,64,60,56,52,48,44,
                                40,36,32,28,24,20,16,12,8,4};


const unsigned char sawtooth[50] = {14,22,30,38,46,54,62,70,78,86,
                                14,22,30,38,46,54,62,70,78,86,
                                14,22,30,38,46,54,62,70,78,86,
                                14,22,30,38,46,54,62,70,78,86,
                                14,22,30,38,46,54,62,70,78,86};

//=============================================================================
//  MAIN
//=============================================================================
void main() {
    C1ON = 0;   // Disable Comparator Module 1
    C2ON = 0;   // Disable Comparator Module 2
    
    PORTA =     0b00000000;     
    TRISA =     0b00000100;     // Switch on A2 now
                                //setup switch (SW1)
    TRISAbits.TRISA2 = 1;       //switch as input
    ANSELAbits.ANSA2 = 0;       //digital switch
    PORTB =     0b00010000;     // 
    TRISB =     0b00100000;     // Port B is not used
    TRISBbits.TRISB5 = 1;       //switch as input
    ANSELBbits.ANSB5 = 1;       //digital switch
    PORTC =     0b00000001;     // 
    TRISC =     0b00000000;     // PORT C, C5 = PWM output, C1 = Cal LED1, C2 = Sine LED2
    
    OSCCON = 0b00110000;        //500KHz clock speed
    //OSCCON = 0b01110000;            // 8MHz Clock Speed
    
    ADCON0 = 0b00101101;                            //select RB5 as source of ADC and enable the module (AN11)
    ADCON1 = 0b00010000;                            //left justified - FOSC/8 speed - Vref is Vdd
    
#ifdef PULL_UPS
    WPUA2 = 1;                  //enable the weak pull-up for the switch
    nWPUEN = 0;                 //enable the global weak pull-up bit
#endif
    
    OPTION_REG = 0b00000111;                    //setup TIMER0
    INTCONbits.TMR0IE = 1;                      //enable the TMR0 rollover interrupt
                                                //setup interrupt on change for the switch
    INTCONbits.IOCIE = 1;                       //enable interrupt on change global
    IOCANbits.IOCAN2 = 1;                       //when SW1 is pressed, enter the ISR
    GIE = 1;                                    //enable global interrupts
    
    T2CON = 0b00000100;		  // TMR2 ON, postscale 1:1, prescale 1:1
    PR2 = 0x80;

    __delay_ms(300);

    pwmstep = 0;
    debounce = 0;

    while(1) {

        if(pwmMode == SINE_WAVE) {
            PORTC = 0b00000001;
        } else if (pwmMode == SQUARE_WAVE) { 
            PORTC = 0b00000010;
        } else if (pwmMode == TRI_WAVE) {
            PORTC = 0b00000100;
        } else if (pwmMode == CAL_WAVE) {
            PORTC = 0b00001000;
        } else {
            pwmMode = 0;
            PORTC = 0b00000001;
        }
        NOP();
        
    if(pwmMode == SINE_WAVE) {   	
        CCP1CON = 0b00001111;   // CCP1 ON, and set to PWM mode
        while(pwmMode == SINE_WAVE) {
            //-------------------------------------------------
            // just sit in loop and load new PWM value every TMR2 cycle
            //-------------------------------------------------
            while(!TMR2IF);            // wait for TMR2 cycle to restart
            CCPR1L = pwmval;           // store PWM value for this cycle
            TMR2IF = 0;                // clear TMR2 int flag

            pwmstep++;                      // inc to next step in sinewave
            if(pwmstep >= 50) pwmstep = 0;  // sine has only 50 steps
                pwmval = sine[pwmstep];         // prepare PWM value ready for next step    
        }
    } else if(pwmMode == SQUARE_WAVE) {
        CCP1CON = 0b00001111;   // CCP1 ON, and set to PWM mode
        while(pwmMode == SQUARE_WAVE){
            CCP1CON = 0b00001111;   // CCP1 ON, and set to PWM mode
            while(pwmMode == SQUARE_WAVE) {
                //-------------------------------------------------
                // just sit in loop and load new PWM value every TMR2 cycle
                //-------------------------------------------------
                while(!TMR2IF);            // wait for TMR2 cycle to restart
                CCPR1L = pwmval;           // store PWM value for this cycle
                TMR2IF = 0;                // clear TMR2 int flag

                pwmstep++;                      // inc to next step in sinewave
                if(pwmstep >= 50) pwmstep = 0;  // sine has only 50 steps
                    pwmval = square[pwmstep];         // prepare PWM value ready for next step    
            }
        }
    } else if(pwmMode == TRI_WAVE) {
        while(pwmMode == TRI_WAVE){
            CCP1CON = 0b00001111;   // CCP1 ON, and set to PWM mode
            while(pwmMode == TRI_WAVE) {
                //-------------------------------------------------
                // just sit in loop and load new PWM value every TMR2 cycle
                //-------------------------------------------------
                while(!TMR2IF);            // wait for TMR2 cycle to restart
                CCPR1L = pwmval;           // store PWM value for this cycle
                TMR2IF = 0;                // clear TMR2 int flag

                pwmstep++;                      // inc to next step in sinewave
                if(pwmstep >= 50) pwmstep = 0;  // sine has only 50 steps
                    pwmval = triangle[pwmstep];         // prepare PWM value ready for next step    
            }
        }
    }
    }
}


void interrupt ISR(void) {
    if (IOCAF) {                                //SW1 was just pressed
        IOCAF = 0;                              //must clear the flag in software
        __delay_ms(300);                          //debounce by waiting and seeing if still held down
        NOP();
        if (SWITCH == DOWN) {
            pwmMode++;
            NOP();
            if(pwmMode > 2) { 
                pwmMode = 0;
            }
        }
    }
    
    if (INTCONbits.T0IF) {
        INTCONbits.T0IF = 0;
        __delay_us(5);
        GO = 1;
        while (GO) continue;
        PR2 = ADRESH;
    }
}