/* sketch for testing AD9850 DDS module (with a 125MHz MCLK)
   with an Arduino Uno or compatible and SPI serial interfacing
   written by Jim Rowe (Silicon Chip)
   Last updated 9/1/2017 at 10:20 am

   Notes:
   1. the following interface connections are assumed by
      the Arduino SPI library:
   SLAVE (AD9850)              ARDUINO (MASTER)
   SS/CS-bar/FQ_UD  <-  from   DigIO pin 10
   DataOut/MOSI     <-  from   DigIO pin 11 (MOSI) or ICSP-4
   DataIn/MISO      ->  to     DigIO pin 12 (MISO) or ICSP-1
   SCK              <-  from   DigIO pin 13 (SCK) or ICSP-3

   2. The SPI interface of the AD9850 DDS device has these
      requirements:
      A. Data on the SDATA/MOSI line is clocked into the AD9850
         on the RISING edge of SCLK pulses.
      B. The serial data must be sent to the AD9850 LSB first -- i.e.,
         in reversed-bit order. So SPI SETTINGS should be set for
         LSB_FIRST. The 32-bit frequency word must also be sent first,
         before the 8-bit control/phase word.
      C. The FQ_UD (freq update/SS) input is active HIGH, and this line
         be kept LOW during the 40 x SCLK pulses used to shift the 
         frequency/control/phase data word into the AD9850. It should then
         be taken HIGH to load the data into the AD9850, no sooner than
         10ns after the last (40th) rising edge of SCK. It should be kept
         high for at least 10ns, before returning it low.
      D. For correct operation, the SPI MODE must be set for 0, ie.,
         CPOL = 0 and CPHA = 0.
   3. To ensure that the AD9850 starts up in serial interface mode, pin2 of
         the device (D2, pin4 of CON1) should be hard wired to GND, while
         pin3 (D1/pin3 of CON1) and pin4 (D0/pin2 of CON1) should be hard wired
         to Vcc/5V (pin1 of CON1).
*/
#include <SPI.h>

// global variables
unsigned int outData1;   // the 1st (lsb) 16 bits of freq data -> AD9850
unsigned int outData2;   // the 2nd (msb) 16 bits of freq data -> AD9850
unsigned int outData3;   // the final 8 bits (control/phase) data -> AD9850

const byte FQUDPin = 10;    // declares IO pin 10 as FQUDPin
const byte MOSIPin = 11;    // and IO pin 11 as MOSIPin
const byte MISOPin = 12;    // and IO pin 12 as MISOPin
const byte SCKPin = 13;     // and finally IO pin 13 as SCKPin

float nuFreq = 100000.00;   // initial value for nuFreq

const float FoFactor = 0.02910383;  // factor for getting FRegVal from
                                    // NuFreq (from serial monitor). Here
                                    // it equals 125,000,000 / 2^32
void setup()
{
  pinMode(FQUDPin, OUTPUT);       // make D10 an output for SS/Load
  digitalWrite(FQUDPin, LOW);       // and initialise it to LOW
  pinMode(MOSIPin, OUTPUT);       // make D11 an output for MOSI data
  digitalWrite(MOSIPin, LOW);     // and initialise to LOW
  pinMode(MISOPin, INPUT);        // make D12 an input for MISO data
                                  // (not used in this sketch)
  pinMode(SCKPin, OUTPUT);        // make D13 an output for SCK
  digitalWrite(SCKPin ,LOW);      // and initialise to LOW
  
  Serial.begin(115200);           // start serial comms at 115200baud, 8N1
   
  SPI.begin();             // start up SPI interface
    
  // now we set the initial values for the AD9850 DDS device
  outData1 = 0x0000;   // resets all registers
  outData2 = 0x0000;
  outData3 = 0x0000;
  SpiWrite();         // sends them all
  // then send the data for an output frequency of 100kHz
  outData1 = 0x6DC6;   // LSB of freq word for 100kHz
  outData2 = 0x0034;   // MSB of freq word for 100kHz
  outData3 = 0x0000;   // byte for normal op, zero phase shift
  SpiWrite();          // DDS should now be set up and producing
                       // a 100kHz sinewave output
  CurrFreqMsg();       // so go advise user, invite a change
}  // end of setup function

// =======================================================================
// start of main loop
// =======================================================================
void loop()
{
  unsigned long FRegVal;       // full 32-bit value to be sent to DDS Freq register
  
  if (Serial.available() > 0)  // see if there any new serial input
  {
    nuFreq = Serial.parseFloat();   // if so, fetch it
    if (nuFreq > 0)             // but only continue if it's not zero
    {
      FRegVal = (nuFreq / FoFactor);    // work out FRegVal
      outData1 = (FRegVal & 0x0000FFFF);        // find its LSB 16-bit half
      outData2 = (FRegVal >> 16) & 0x0000FFFF;  // and its MSB 16-bit half
      outData3 = 0x0000;                    // (phase unchanged)
      SpiWrite();                     // now send off to AD9850 DDS chip
      CurrFreqMsg();                  // send details back to PC
    }
  }
}   // end of main loop

// =====================================================================
// SpiWrite() function: to send 16 bits of data to the AD9833 via SPI
// =====================================================================

void SpiWrite()
{
  digitalWrite(SCKPin,LOW);      // make sure SCK pin is high
  digitalWrite(MOSIPin,LOW);     // and the MOSI pin low
  digitalWrite(FQUDPin,LOW);       // then take the FQUD pin LOW as well
  
  // first set the SPI port for 5MHz clock, data sent LSB first, and
  // data mode 0 -- with SCK idle low (CPOL = 0) and
  //                MOSI data read on rising edge (CPHA = 0)
  SPI.beginTransaction(SPISettings(5000000, LSBFIRST, SPI_MODE0));
  
  // then begin the SPI transaction
  long recVal16;          // local variable for SPI received data (not used)
  recVal16 = SPI.transfer16(outData1);  // send 1st 16 bits of data out
  recVal16 = SPI.transfer16(outData2);  // followed by 2nd 16 bits
  recVal16 = SPI.transfer(outData3);    // and finally the 8 cont/phase bits
  SPI.endTransaction();
  digitalWrite(FQUDPin, HIGH); // pull FQUD pin HIGH for loading in data  
  digitalWrite(FQUDPin, LOW);     // then bring it low again
  digitalWrite(MOSIPin,LOW);     // also drop the MOSI pin
}  // end of SpiWrite() function

// ======================================================================
// CurrFreqMsg() function: to send information to serial monitor on PC
// showing current frequency, also request desired new frequency
// ======================================================================

void CurrFreqMsg()
{
  String strUnits;    // string for Hz, kHz or MHz suffix
  String strFreqPrint;  // string for printing out current frequency
  long CurrFreq = long(nuFreq); // first get CurrFreq for testing
  
  if (CurrFreq <1000)
  {
    strUnits = " Hz"; 
  }
  else if (CurrFreq >= 1000 & CurrFreq < 1000000)
  {
    strUnits = " kHz";
    nuFreq = nuFreq / 1000;
  }
  else if (CurrFreq >= 1000000)
  {
    strUnits = " MHz";
    nuFreq = nuFreq / 1000000;
  }
  strFreqPrint = "Current Frequency = " + String(nuFreq) + strUnits;
  Serial.println(strFreqPrint);
  Serial.println("Type in required new frequency (Hz): ");

}   // end of CurrFreqMsg() function
// =====================================================================




