/*
    Module: sdTable.c

    Author: Mike Hibbett

    Date: 21/11/09

    Function: Here lives the implementations of debug routines. You don't have to place
              them here; application specific ones may well be implemented in your own
              code, it's up to you.

              Refer to serialDebug.h for more details.

*/

#include <stdio.h>
#include <string.h>
#include <usart.h>
#include <ctype.h>
#include "sdTable.h"


void sdFlash( char * );
void sdHelp( char * );
unsigned char getArg16(char **, unsigned int *);


cmdList_t cmdTable[] = 	{
						{ "flash",  sdFlash,   "Displays contents of flash" },
						{ "help",   sdHelp,    "Displays this screen" },
						{ NULL,     NULL,   NULL   } /* End of table marker - do not remove */
						};


/*****************************************************************************
    Function: getArg16

    Purpose:  extracts the next argument from args, converting it to an unsigned int.

    Returns:  0 if integer argument not found, 1 for success.
              Input pointer advanced on success to first unused character
*****************************************************************************/
unsigned char getArg16( char **args, unsigned int *val )
{
    unsigned char retVal = 0;
    char *ptr = *args;
    
    *val = 0;

    /* Skip any possible invalid characters */
    while (!isalnum(*ptr))
        ptr++;
    
    /* What radix is the number? We support 10 or 16 */
    if ( (*ptr == '0') && ( tolower(*(ptr+1)) == 'x' ) ) {
        
        /* Hexidecimal */
        ptr+=2; 
        
        if ( isxdigit(*ptr) ) {
        
            while ( isxdigit(*ptr) ) {
                
                *val = *val * 16;
                
                if ( *ptr <= '9') {
                    
                    *ptr = *ptr - '0';
                } else {
                
                    *ptr = (*ptr | 0x20) - 'a' + 10;
                }
                
                *val += *ptr++;     
            }
            
            retVal = 1;
            /* Move the original argument forward */
            *args = ptr;
        }        
    } else {
    
        /* Decimal */
        while ( isdigit(*ptr) ) {
            *val = *val * 10;
            *val += *ptr++ - '0';
        }
        
        retVal = 1;
        /* Move the original argument forward */
        *args = ptr;
    }    
    
    return retVal;
}

    
    
/*****************************************************************************
    Function: put8x

    Purpose:  Puts a 8 bit unsigned char out of serial port as a 2.2X format.
              This saves using printf, which is *large*!

    Returns:  void
*****************************************************************************/
void put8x( unsigned char val )
{
    unsigned char hex[] = "0123456789ABCDEF";  
    char str[3];
    unsigned char cval;
    
    str[2] = 0;
    cval = val % 16;
    str[1] = hex[cval];
    val >>= 4;
    str[0] = hex[val];
    putsUSART(str);
}



/*****************************************************************************
    Function: put16x

    Purpose:  Puts a 16 bit unsigned int out of serial port as a 4.4X format.
              This saves using printf, which is *large*!

    Returns:  void
*****************************************************************************/
void put16x( unsigned int val )
{
    unsigned char hex[] = "0123456789ABCDEF";  
    char str[5];
    unsigned char cval;
    
    cval = val % 16;
    str[4] = 0;
    str[3] = hex[cval];
    val >>= 4;
    cval = val % 16;
    str[2] = hex[cval];
    val >>= 4;
    cval = val % 16;
    str[1] = hex[cval];
    val >>= 4;
    str[0] = hex[val];
    putsUSART(str);
}


/*****************************************************************************
    Function: putDec

    Purpose:  Puts a decimal number over the serial port.
              This saves using printf, which is *large*!

    Returns:  void
*****************************************************************************/
void putDec( unsigned int val )
{
    char str1[16];
    char str2[16];
    unsigned char lp;
    unsigned char len;
    
    lp = 0;
    
    do {
        str1[lp++] = '0' + (val % 10);
        val = val / 10;        
    } while ( val );
    
    str1[lp] = 0;
    str2[lp] = 0;
    
    /* Now reverse the string */
    len = strlen(str1);
    for ( lp=0; lp < len; lp++ ) {
        str2[len - lp-1] = str1[lp];
    }    
    
    putsUSART(str2);
}


    
/*****************************************************************************
    Function: sdHelp

    Purpose:  dumps the contents of ram.
              args format: address length
              where address is 16 bit and length is 8 bit, 0x or decimal format

    Returns:  None
*****************************************************************************/
void sdHelp( char *args )
{
    cmdList_t *sdp = cmdTable;
    
    putrsUSART((const far rom char *)"Debug commands are:\r\n");
    
    while ( sdp->name != NULL ) {
        while (BusyUSART());
        putcUSART('\t');
        putrsUSART((const far rom char *)sdp->name);
        while (BusyUSART());
        putcUSART('\t');
        putrsUSART((const far rom char *)sdp->description);
        putrsUSART((far rom char *)"\r\n");
        
        sdp++;
    }
}


/*****************************************************************************
    Function: sdFlash

    Purpose:  dumps the contents of flash.
              args format: address length
              where address is 16 bit and length is 8 bit, 0x or decimal format

    Returns:  None
*****************************************************************************/
void sdFlash( char *args )
{
    unsigned int add, num, lp, lp2;
	rom unsigned char *ptr;
	char ch;
	unsigned int startAdd, endAdd;
   
    if ( !getArg16(&args, &add )) {
        putrsUSART((const far rom char *)"Typo entered in address.");
        
    } 
	if ( !getArg16(&args, &num )) {
        putrsUSART((const far rom char *)"Typo entered in length.");
        
    } else {
    
        startAdd = add & 0xFFF0;
        endAdd = (num & 0xFFF0) + 16;
    
		ptr = ( rom unsigned char *)startAdd;

		putrsUSART((const far rom char *)"Addr 000102030405060708090A0B0C0D0E0F ASCII\r\n");

        // display contents of flash 
		for ( lp=0; lp<endAdd; lp+=16 ) {
			put16x(startAdd);
            while (BusyUSART());
			putcUSART(' ');
			
    		for (lp2=0; lp2 < 16; lp2++) {
    			put8x(*ptr);
    			startAdd++;
    			ptr++;
            }
            
	        while (BusyUSART());
    		putcUSART(' ');
			
			ptr -=16;
			startAdd -=16;
			
    		for (lp2=0; lp2 < 16; lp2++) {
    			if ( (*ptr >= ' ') && ( *ptr <= '~') ) {
                    while (BusyUSART());
    			    putcUSART(*ptr);
    			} else {
                    while (BusyUSART());
    			    putcUSART('.');
    			}
    			startAdd++;
    			ptr++;
            }
            
            while (BusyUSART());
 			putcUSART('\r');
			while (BusyUSART());
			putcUSART('\n');			
		}
    }
}


