// The following routines are modified versions of the C18 library
// LCD routines.  They were modified because I had difficulty in 
// getting the generic routines to work with a 4-bit interface using
// the upper nibble for data transfer.  
// Subroutine names were prefixed with an "X" to prevent confusion with
// generic library routines.
// R Lang 12/2005
//   The PIC pin definitions used below are defined in Xxlcd.h.
//   DATA LINES = RB4-7
//   RS PIN= PIC RA2 to LCD connector Pin4
//   RW PIN= PIC RA3 to LCD connector Pin5
//   E PIN= PIC RA5 to LCD connector Pin7
#include <p18cxxx.h>
#include "xxlcd.h"
/********************************************************************
*       Function Name:  XOpenXLCD                                    *
*       Return Value:   void                                        *
*       Parameters:     lcdtype: sets the type of LCD (lines)       *
*       Description:    This routine configures the LCD. Based on   *
*                       the Hitachi HD44780 LCD controller. The     *
*                       routine will configure the I/O pins of the  *
*                       microcontroller, setup the LCD for 4-bit
*                       mode and clear the display. The user  *
*                       must provide three delay routines:          *
*                       DelayFor18TCY() provides a 18 Tcy delay     *
*                       DelayPORXLCD() provides at least 15ms delay *
*                       DelayXLCD() provides at least 5ms delay     *
********************************************************************/
void XOpenXLCD(void )
{        unsigned char i;   
      // Upper 4-bits of the port
        // SET UP B PORT 7-4 OUTPUT FOR LCD
        //               3-2 INPUT FOR SWITCHES
        //               0 OUTPUT FOR LCD POWER CONTROL
        TRIS_DATA_PORT =0b00001110;

        TRIS_RW = 0;                    // All control signals made outputs
        TRIS_RS = 0;
        TRIS_E = 0;
        RW_PIN = 0;                     // R/W pin made low
        RS_PIN = 0;                     // Register select pin made low
        E_PIN = 0;                      // Clock pin made low

        // Delay for 15ms to allow for LCD Power on reset
        DelayPORXLCD();
        
        // Setup interface to LCD
        DATA_PORT &= 0x0f;
        DATA_PORT |= 0b00110000;  
        DelayFor18TCY();
        for (i=1;i<=3;i++)               // DO 3 TIMES TO INIT
        {
        E_PIN = 1;                      // Clock the cmd in
        DelayFor18TCY();
        E_PIN = 0;
        // Delay for at least 4.1ms
        DelayXLCD();
        }

        // Setup interface to LCD
        DATA_PORT &= 0x0f;
        DATA_PORT |= 0b00100000;  // Function set cmd(4-bit interface)
        DelayFor18TCY();
        E_PIN = 1;                      // Clock the cmd in
        DelayFor18TCY();
        E_PIN = 0;
        // Delay for at least 4.1ms
        DelayXLCD();  


// FIRST DO FUNCTION SET TO SET INTERFACE LENGTH, #LINES, CHAR FONT
        XWriteCmdXLCD(0b00101000);// Function set cmd 4-bit interface, multiple lines,5x8 char 
// DISPLAY CONTROL
        while(XBusyXLCD());          // Wait if LCD busy
       XWriteCmdXLCD(0b00001000);    // Display OFF/Blink OFF

        while(XBusyXLCD());          // Wait if LCD busy
        XWriteCmdXLCD(0b00001111);   // Display ON/Cursor ON/BLINK ON

        while(XBusyXLCD());              // Wait if LCD busy
        XWriteCmdXLCD(0b00000001);             // Clear display

        while(XBusyXLCD());              // Wait if LCD busy
        XWriteCmdXLCD(0b00000110);   // Entry Mode inc, no shift

        while(XBusyXLCD());  // Wait if LCD busy
        XSetDDRamAddr(0);   // Set Display data ram address to 0, home
        return;
}
/********************************************************************
*       Function Name:  XputrsXLCD
*       Return Value:   void
*       Parameters:     buffer: pointer to string
*       Description:    This routine writes a string of bytes to the
*                       Hitachi HD44780 LCD controller. The user
*                       must check to see if the LCD controller is
*                       busy before calling this routine. The data
*                       is written to the character generator RAM or
*                       the display data RAM depending on what the
*                       previous SetxxRamAddr routine was called.
********************************************************************/
void XputrsXLCD(const rom far char *buffer)
{
        while(*buffer)                  // Write data to LCD up to null
        {
                while(XBusyXLCD());      // Wait while LCD is busy
                XWriteDataXLCD(*buffer); // Write character to LCD
                buffer++;               // Increment buffer
        }
        return;
}

/********************************************************************
*       Function Name:  XputsXLCD
*       Return Value:   void
*       Parameters:     buffer: pointer to string
*       Description:    This routine writes a string of bytes to the
*                       Hitachi HD44780 LCD controller. The user
*                       must check to see if the LCD controller is
*                       busy before calling this routine. The data
*                       is written to the character generator RAM or
*                       the display data RAM depending on what the
*                       previous SetxxRamAddr routine was called.
********************************************************************/
void XputsXLCD(char *buffer)
{
        while(*buffer)                  // Write data to LCD up to null
        {
                while(XBusyXLCD());      // Wait while LCD is busy
                XWriteDataXLCD(*buffer); // Write character to LCD
                buffer++;               // Increment buffer
        }
        return;
}

/********************************************************************
*       Function Name:  XSetDDRamAddr                                *
*       Return Value:   void                                        *
*       Parameters:     CGaddr: display data address                *
*       Description:    This routine sets the display data address  *
*                       of the Hitachi HD44780 LCD controller. The  *
*                       user must check to see if the LCD controller*
*                       is busy before calling this routine.        *
********************************************************************/
void XSetDDRamAddr(unsigned char DDaddr)
{
	    while(XBusyXLCD());  
 // Upper nibble  interface
        TRIS_DATA_PORT &= 0x0f;                 // Make port output
        DATA_PORT &= 0x0f;                      // and write upper nibble
        DATA_PORT |= ((DDaddr | 0b10000000) & 0xf0);
        RW_PIN = 0;                             // Set control bits
        RS_PIN = 0;
        DelayFor18TCY();
        E_PIN = 1;                              // Clock the cmd and address in
        DelayFor18TCY();
        E_PIN = 0;
  // Upper nibble interface
        DATA_PORT &= 0x0f;                      // Write lower nibble
        DATA_PORT |= ((DDaddr<<4)&0xf0);
        DelayFor18TCY();
        E_PIN = 1;                              // Clock the cmd and address in
        DelayFor18TCY();
        E_PIN = 0;
             // Upper nibble interface
        TRIS_DATA_PORT |= 0xf0;                 // Make port input
        return;
}

/********************************************************************
*       Function Name:  XWriteCmdXLCD                                *
*       Return Value:   void                                        *
*       Parameters:     cmd: command to send to LCD                 *
*       Description:    This routine writes a command to the Hitachi*
*                       HD44780 LCD controller. The user must check *
*                       to see if the LCD controller is busy before *
*                       calling this routine.                       *
********************************************************************/
void XWriteCmdXLCD(unsigned char cmd)
{
       // Upper nibble interface
        TRIS_DATA_PORT &= 0x0f; // MAKE UPPER 4 BITS OUTPUT
        DATA_PORT &= 0x0f;  // CLEAR UPPER BITS AND KEEP LOWER UNCHANGED
        DATA_PORT |= cmd&0xf0; // PUT UPPER FOUR BITS IN DATAPORT

        RW_PIN = 0;                     // Set control signals for command
        RS_PIN = 0;
        DelayFor18TCY();
        E_PIN = 1;                      // Clock command in
        DelayFor18TCY(); 
        E_PIN = 0;
        
        DATA_PORT &= 0x0f; // CLEAR UPPER 4 BITS AND KEEP LOWER UNCHANGED
        DATA_PORT |= (cmd<<4)&0xf0; // LOAD LOWER 4 BITS OF cmd INTO UPPER 4 BITS OF DATAPORT
        DelayFor18TCY();
        E_PIN = 1;                      // Clock command in
        DelayFor18TCY();
        E_PIN = 0;
        TRIS_DATA_PORT |= 0xf0;  // SET DATAPORT FOR INPUT

        return;
}

/********************************************************************
*       Function Name:  XWriteDataXLCD                               *
*       Return Value:   void                                        *
*       Parameters:     data: data byte to be written to LCD        *
*       Description:    This routine writes a data byte to the      *
*                       Hitachi HD44780 LCD controller. The user    *
*                       must check to see if the LCD controller is  *
*                       busy before calling this routine. The data  *
*                       is written to the character generator RAM or*
*                       the display data RAM depending on what the  *
*                       previous SetxxRamAddr routine was called.   *
********************************************************************/
void XWriteDataXLCD(char data)
{
   // Upper nibble interface
        TRIS_DATA_PORT &= 0x0f;
        DATA_PORT &= 0x0f;
        DATA_PORT |= data&0xf0;
        RS_PIN = 1;                     // Set control bits
        RW_PIN = 0;
        DelayFor18TCY();
        E_PIN = 1;                      // Clock nibble into LCD
        DelayFor18TCY();
        E_PIN = 0;
                     // Upper nibble interface
        DATA_PORT &= 0x0f;
        DATA_PORT |= ((data<<4)&0xf0);
        DelayFor18TCY();
        E_PIN = 1;                      // Clock nibble into LCD
        DelayFor18TCY();
        E_PIN = 0;
                    // Upper nibble interface
        TRIS_DATA_PORT |= 0xf0;
        return;
}
/********************************************************************
*       Function Name:  XBusyXLCD                                    *
*       Return Value:   char: busy status of LCD controller         *
*       Parameters:     void                                        *
*       Description:    This routine reads the busy status of the   *
*                       Hitachi HD44780 LCD controller.             *
********************************************************************/
unsigned char XBusyXLCD(void)
{       RW_PIN = 1;      // Set the control bits for read
        RS_PIN = 0;
        DelayFor18TCY();
        E_PIN = 1;             // Clock in the command
        DelayFor18TCY();
//******************************************************        
// REMOVE FOLLOWING LINE WHEN LCD IS INSTALLED        
// XWriteCmdXLCD(0x00);  // NEVER RETURN BUSY (DEBUG ONLY)
//*******************************************************        
  // Upper nibble interface
        if(DATA_PORT&0x80)

        {
                E_PIN = 0;              // Reset clock line
                E_PIN = 1;              // Clock out lower nibble
                DelayFor18TCY();
                E_PIN = 0;
                RW_PIN = 0;             // Reset control line
                return 1;               // Return TRUE
        }
        else                            // Busy bit is low
        {
                E_PIN = 0;              // Reset clock line
                E_PIN = 1;              // Clock out lower nibble
                DelayFor18TCY();
                E_PIN = 0;
                RW_PIN = 0;             // Reset control line
                return 0;               // Return FALSE
        }
}


