/*
 * File:   i2c.c
 * Author: mike
 *
 * Created on 31 March 2019, 19:27
 * 
 * Functions for accessing the I2C bus on a PIC10F202
 */


#include <xc.h>

#define SCL_HIGH do {TRIS_buff |= 0x01; TRISGPIO = TRIS_buff;} while (0)
#define SCL_LOW  do {TRIS_buff &= ~0x01; TRISGPIO = TRIS_buff;} while (0)
#define SDA_HIGH do {TRIS_buff |= 0x02; TRISGPIO = TRIS_buff;} while (0)
#define SDA_LOW  do {TRIS_buff &= ~0x02; TRISGPIO = TRIS_buff;} while (0)
#define SCL_IN GPIObits.GP0
#define SDA_IN GPIObits.GP1

// We need a copy of the TRIS register
// as it is a write only register
unsigned char TRIS_buff;


void i2c_init(void)
{    // Initialise the I2C bus
    TRIS_buff = 0x03;
    TRISGPIO = TRIS_buff;
}


void i2c_start(void)
{
    SDA_HIGH;
    __nop();
    SCL_HIGH;
    __nop();
    SDA_LOW;
    __nop();
    SCL_LOW;
    __nop();    
}


void i2c_stop(void)
{
    SDA_LOW;
    __nop();
    SCL_HIGH;
    __nop();
    SDA_HIGH;
    __nop();
}


unsigned char i2c_rx(unsigned char ACK)
{
    unsigned char lp;
    unsigned char rx_val;
    
    SDA_HIGH;
  
    for (lp = 0; lp < 8; lp++) {
        rx_val <<= 1;
        
        // Set SCL HIGH, check for device stretching
        // the clock line low indicating it is busy
        do {
          SCL_HIGH;
          __nop();
        } while(SCL_IN==0); 
        
        if (SDA_IN) 
            rx_val |= 1;
        
        SCL_LOW;
        __nop();
    } 

    // send the ACK bit
    if (ACK) 
        SDA_LOW;
    else 
        SDA_HIGH;
    
    SCL_HIGH;
    __nop();
    SCL_LOW;
    SDA_HIGH;
  
    return rx_val;
}


unsigned char i2c_tx(unsigned char tx_val)
{
    unsigned char lp, ACK;

    for (lp = 8; lp; lp--) {
        if (tx_val & 0x80) 
            SDA_HIGH;
        else 
            SDA_LOW;
    
        __nop();
        SCL_HIGH;
        tx_val <<= 1;
        SCL_LOW;
        __nop();
    }
    
    SDA_HIGH;
    SCL_HIGH;
    __nop();
    
    ACK = SDA_IN;   
    
    SCL_LOW;
    
    return ACK;
}
