/*
  Altitude 3
  
  From the PicNMix column in Everyday Practical Electronics, May 2012.
  Mike Hibbett 
  
  The extent of the display is 
  2,0 to 131,129
  
  Format x,y where x goes down, away from the connector
  y goes from left to right ( when the connector is at the top )
  
 */

#include "lcd_epson.h"
#include "bmp.h"
 
// Application constants 
const unsigned int BUTTON_PIN   = 37;  // Pin to which the button is connected

// Application global variables
char altStr[30];          // Holds the displayable altitude information
char satString[4];        // Holds the number of satellites used, as a string
char metersString[16];    // String to hold the raw altitude text from the GPS
unsigned int commaCount;  // used to parse the GPGGA sentence
unsigned int RXState = 0; // serial processing state, used to track position in GPS data
unsigned int strIndex = 0;// used by the GPS decode routine to store data in the meterString variable
unsigned long startTime;  // Counts 5.5s intervals for display update
int altPoints[128];       // historic data points for alititude graph.
int minAlt;               // The minimum altitude in the range of values
int maxAlt;               // The maximum altitude in the range of values
int rangeAlt;             // The range ( min to max )
int offsetAlt;            // Value to offset the dataset by, so the smallest value is zero
float divAlt;             // The multiplier to get the 
int altVal;               // Integer representation of the altitude


void setup() 
{ 
  int lp;  // Loop counter
  
  // Configure the button pin as an input.
  pinMode(BUTTON_PIN, INPUT);
  
  // Configure the GPS serial port
  Serial1.begin(4800);
  
  // Setup the LCD pin port directions, and initialise the display
  LCDSetup();

  // Initialise the display strings and other global variables
  altStr[0] = '\0'; metersString[0] = '\0'; startTime = 0;
  for ( lp=0; lp < 128; lp++ ) altPoints[lp] = 0;
  
  // Put the logo and other text on the display  
  LCDPutBMP(bitmap,2,0, image_width, image_height);
  
  // Draw the box for the graph
  LCDSetLine(89,0, 130, 0, GREEN);
  LCDSetLine(89,129, 130, 129, GREEN);
  LCDSetLine(89,0, 89, 129, GREEN);
  LCDSetLine(130,0, 130, 129, GREEN);
}


void loop() 
{  
  getAltitude();  // Updates "metersString" from GPS data
  
  // Every 5.52, update teh display
  if ( (millis() - startTime) > 5500 ) {
    startTime = millis();
    
    // get the altitude as a number, for use on the graph
    altVal = atoi(metersString);
    
    // Remove the last text displayed, and write the new one
    LCDPutStr(altStr,40,3,VLARGE,BLACK,BLACK);      
    if ( metersString[0] == '\0' ) {
      sprintf(altStr, "No GPS", metersString);
    } else {
      sprintf(altStr, "%sM", metersString);
    }
    LCDPutStr(altStr,40,3,VLARGE,WHITE,BLACK);
    
    // Update the progress graphic display
    calcProgress();
    displayProgress();    
  }
}


// Calculates the offset and multipler factors for the graph
void calcProgress( void )
{
  int lp;
  minAlt = 30000;  // These two allow us to calc the maximum & min values
  maxAlt = -30000; // in the altitude data array
  
  // Shift the datapoints to the left, calculating max/min as we go.
  for ( lp=0; lp < 127; lp++ ) {
    altPoints[lp] = altPoints[lp+1];
    if ( altPoints[lp] < minAlt ) minAlt = altPoints[lp];
    if ( altPoints[lp] > maxAlt ) maxAlt = altPoints[lp];
  }
  altPoints[127] = altVal;  // add in the most recent value
  if ( altVal < minAlt ) minAlt = altVal;
  if ( altVal > maxAlt ) maxAlt = altVal;
  
  // Now calculate the scaling value and offset, to display the points correctly
  rangeAlt = maxAlt - minAlt;
  offsetAlt = -minAlt;
  divAlt = (float)rangeAlt / 39.0;    
}

// Display the graph of the altitude points. You must call calcProgress first
void displayProgress( void )
{
  int lp;
  int val;
  
  for ( lp=0; lp < 128; lp++ ) {
    val = (altPoints[lp] + offsetAlt) / divAlt; 
    
    // This simultaenously displays the new points & clears the previous graph
    LCDSetLine(129,lp+1, 129-val, lp+1, BLUE);  
    LCDSetLine(129-val,lp+1, 90, lp+1, BLACK);  
  }    
}


// Call instead of getAltitude to output the GPS data onto the display.
// Pressing the button will pause the output
int getAltitudeDebug( void )
{
  int x,y;
  char inByte;
  
  x=0;y=0;
  LCDCLS( BLACK );
  
  do {
    
    if (Serial1.available() > 0) {
      // get incoming byte:
      inByte = Serial1.read();  

      if ( digitalRead(BUTTON_PIN) == HIGH ) {
        LCDPutChar(inByte, x, y, SMALL, WHITE, BLACK);
        y = y + 6;
        
        if ( y > 130 ) {
          y = 0;
          x = x + 9;
          
          if ( x > 130 ) {
            x = 0;
            LCDCLS( BLACK );
          }
        }
      }
    }    
  } while (1);
}


// This is a statemachine ( or "FSM" ) which can be called as frequently as you like.
// When a character from the GPS module is detected, it updates the receive state ( RXState )
// and performs any action needed.
// An FSM is used here as the format of the incoming data is fixed. It's also very
// efficient, as only a few lines of code get executed each time it is called.
void getAltitude( void )
{
  char inByte;
    
  if (Serial1.available() > 0) {
    // get incoming byte:
    inByte = Serial1.read();  
    
    switch (RXState) {
      case 0: 
        if (inByte == 'G') RXState++;
        break;
      case 1:
        if (inByte == 'P') RXState++; else RXState = 0;
        break;
      case 2:
        if (inByte == 'G') RXState++; else RXState = 0;
        break;
      case 3:
        if (inByte == 'G') RXState++; else RXState = 0;
        break;
      case 4:
        if (inByte == 'A') {
          RXState++; 
          commaCount = 0;
        } else RXState = 0;
        break;
      case 5:
        /* Wait for the six commas */
        if (inByte == ',') {
          commaCount++;
          if ( commaCount == 6 ) RXState++;
        }
        break;
      case 6:
        /* this is our byte */
        if (inByte == '0') {
          // GPS Signal invalid. No point looking for altitude data.
          // Return to the beginning, to wait for the next sentence
          metersString[0] = '\0';
          RXState = 0;
        } else {
          /* GPS Signal good */
          RXState++; 
        }
        break;
      case 7:
        // skip the comma
        RXState++;
        strIndex = 0;
        break;      
      case 8:
        // Read off the number of satellites, just in case it is useful
        if (inByte != ',') {      
          satString[strIndex++] = inByte;  
        } else {
          satString[strIndex] = '\0';
          RXState++;
        }
        break;
      case 9:
        /* Wait for another comma to get to the altitude text */
        if (inByte == ',') {
            RXState++;
            strIndex = 0;
        }
        break;
      case 10:
        // start to copy text out of the message
        if (inByte != ',') {      
          metersString[strIndex++] = inByte;  
        } else {
          metersString[strIndex] = '\0';
          RXState = 0;
          strIndex = 0;
        }
        break;      
    }
  }   
}


