
#include "FSIO.h"

// Configure config bits
_FOSCSEL(FNOSC_FRC);                              // FRC Oscillator
_FOSC(FCKSM_CSECMD & OSCIOFNC_OFF & POSCMD_XT);
                                                  // Clock Switching is enabled and Fail Safe Clock Monitor is disabled
                                                  // OSC2 Pin Function: OSC2 is Clock Output

                                                  // Primary Oscillator Mode: Disabled
// No Watchdog
_FWDT(FWDTEN_OFF);                                // Watchdog Timer Enabled/disabled by user software
_FICD(ICS_PGD1 & JTAGEN_OFF)                      // No jtag. This makes some I/O pins available

/**********************************************************************
* Symbolic Constants
**********************************************************************/
#define DMA_BUFF_SIZE    (256)          // Size of the DAC DMA buffer; two per DAC
#define CHUNK_ID_IN_HEX  (0x61746164)   // Representation of "data" chunkid as a 32 bit hex value
#define CACHE_BUFF_SIZE  (1024)         // Size of the cache in bytes. Holds the combined data for a buffer pair

/**********************************************************************
* Global Variables
**********************************************************************/

char receiveBuffer[16];
unsigned char cacheBuff[CACHE_BUFF_SIZE];

unsigned int RightBufferA[DMA_BUFF_SIZE] __attribute__ ((space(dma)));
unsigned int RightBufferB[DMA_BUFF_SIZE] __attribute__ ((space(dma)));
unsigned int LeftBufferA[DMA_BUFF_SIZE]  __attribute__ ((space(dma)));
unsigned int LeftBufferB[DMA_BUFF_SIZE]  __attribute__ ((space(dma)));

unsigned char rightEmpty = FALSE;  // Indicates when a right channel dma buffer is empty
unsigned char leftEmpty = FALSE;   // Indicates when a left channel dma buffer is empty



/**********************************************************************
* Local Function Prototypes
**********************************************************************/
void Init_DAC_DMA( void );
unsigned char load_buffers( FSFILE *pointer, unsigned long *bytesRemaining, unsigned char start);
    
    
int main (void)
{
     FSFILE * pointer;
     unsigned long chunkSize,chunkID, offset, fileDataSize;
     unsigned char moreData=TRUE;

     // Configure Oscillator to operate the device at (very nearly) 40MIPS
     // Fosc= 78749990Hz with 3.5 something MHz xtal
     PLLFBD=86;               // M=88
     CLKDIVbits.PLLPOST=0;    // N1=2
     CLKDIVbits.PLLPRE=0;     // N2=2
     OSCTUN=0;                // Tune FRC oscillator, if FRC is used

     // Disable Watch Dog Timer
     RCONbits.SWDTEN=0;
     
     __builtin_write_OSCCONH(0x03);
     __builtin_write_OSCCONL(0x01);  
     // Wait for Clock switch to occur
     while (OSCCONbits.COSC != 0b011);
     // Wait for PLL to lock
     while(OSCCONbits.LOCK!=1) {};   
     
     // Map the SPI Bus pins: 
     // SDI1 input to RP9
     RPINR20bits.SDI1R = 9;
     // SDO1 output to RP8
     RPOR4bits.RP8R = 7;
     // SCK1 output to RP7
     RPOR3bits.RP7R = 8;
        
     while (!MDD_MediaDetect());

     // Initialize the library
     while (!FSInit());

	 // Open our wav file on the SDMedia card
     pointer = FSfopen ("SOUND.WAV", "r");
    
     // If we could not find the file, just hang here 
     if (pointer == NULL)
          while(1);
    
     // Read the initial Chunk header, and discard.
     // If the file isn't a wav file it is the user's fault, and we will just pause at this point
     if (FSfread (receiveBuffer, 12, 1, pointer) != 1)
          while(1);  
     
     // skip the initial header
     offset = 12;   
      
     do {         
          // A bit of cunning. Read the chunkID as a long, and check the value is 
          // 0x61746164
          if (FSfread (&chunkID, 4, 1, pointer) != 1)
               while(1);
          offset+= 4;
            
          // Read the subchunk size
          if (FSfread (&chunkSize, 4, 1, pointer) != 1)
               while(1);
          offset+= 4;
            
          // Check if this is the correct subchunk
          if ( chunkID != CHUNK_ID_IN_HEX ) {
               // Move to the start of the next chunk
               offset += chunkSize;
               FSfseek(pointer, offset, SEEK_SET);   
          }    
     } while (chunkID != 0x61746164);
    

     // This is the main play loop, where we wait for the button to be pressed
     do {          
          fileDataSize = chunkSize;
          FSfseek(pointer, offset, SEEK_SET);
          
          // The file pointer, and offset, point to the start of the data.
          // chunkSize is how many bytes of data need to be read.
        
          // Fill the left and right buffer A, then buffer B - safe to assume we have enough data at start.
          // This is only 5ms worth of data. 
          
          // Load the "A" buffers...
          load_buffers( pointer, &fileDataSize, 1 );
          // and then the "B" buffers
          load_buffers( pointer, &fileDataSize, 0 );
          
          
          // Initialise DMA and DAC systems
          leftEmpty=FALSE; rightEmpty=FALSE;
          moreData = TRUE;
          DMA0CONbits.CHEN = 0;
          DMA1CONbits.CHEN = 0;
          IEC0bits.DMA0IE = 0; 
          IEC0bits.DMA1IE = 0; 
          
	      // This would be a good place to test for the switch being pressed, and then released.
          // for now, the sound file will start as soon as the unit powers up.  
        
          // Start the DAC running, from the DMA system.
          Init_DAC_DMA();
          
         
          // Feed the DMA buffers with data.
          do {
               // Wait for both the DMA buffers ( left & right ) to empty.
               if ( (leftEmpty==TRUE) && (rightEmpty==TRUE) ) {
                    leftEmpty=FALSE;
                    rightEmpty=FALSE;
                    // Go load the buffers
                    moreData = load_buffers( pointer, &fileDataSize, 0 );
               }  
                  
          } while ( moreData );
        
          while ( (leftEmpty==FALSE) && (rightEmpty==FALSE) ) {
               // Wait for current buffer to empty    
          } 
          
          leftEmpty=FALSE;
          rightEmpty=FALSE;
               
          while ( (leftEmpty==FALSE) && (rightEmpty==FALSE) ) {
               // Wait for final buffer to empty    
          } 
     
          leftEmpty=FALSE;
          rightEmpty=FALSE;
         
          // Turn off DMA reload. 
          DMA0CONbits.CHEN = 0;   /* Disable DMA Channel 0 */
          DMA1CONbits.CHEN = 0;   /* Disable DMA Channel 1 */
          DAC1CONbits.DACEN = 0;  /* Disable DAC1 Module */
    
     } while (1);    
}


/**********************************************************************
* Local Functions
**********************************************************************/


// Maintains knowledge of which buffer to fill next, starting with buffer A.
// Data format is 4 bytes at a time
// AABBCCDD
// 0xBBAA = left; 0xDDCC = right. 2's COMPLIMENT SIGNED
unsigned char load_buffers( FSFILE *pointer, unsigned long *bytesRemaining, unsigned char start )
{
     static unsigned char fillBufferA = TRUE;     // Which buffer to fill next
     register int lp;                             // Loop counter
     unsigned char moreData = TRUE;               // Indicates when we have more data to read
     
     if ( start ) {
          fillBufferA = TRUE;
     }
          
     if ( *bytesRemaining >= CACHE_BUFF_SIZE ) {
          if (  FSfread (cacheBuff, CACHE_BUFF_SIZE, 1, pointer) != 1 ) {
               while(1);    
          }     
          *bytesRemaining -= CACHE_BUFF_SIZE;           
     } else {
          if ( *bytesRemaining > 0 ) {
               FSfread (cacheBuff, *bytesRemaining, 1, pointer);    
               moreData = FALSE;       
          }
          memset(cacheBuff, 0, CACHE_BUFF_SIZE- *bytesRemaining); 
          *bytesRemaining = 0;
     }

     if ( fillBufferA ) {
          for ( lp=0; lp < DMA_BUFF_SIZE; lp++ ) {
               // Fill BufferA's left and then right samples  
               memcpy(&LeftBufferA[lp], &cacheBuff[lp*4], 2);
               memcpy(&RightBufferA[lp], &cacheBuff[2 + (lp*4)], 2);
          } 
                    
          fillBufferA = FALSE;               
     } else {
          for ( lp=0; lp < DMA_BUFF_SIZE; lp++ ) {
               // Fill BufferB's left and then right samples    
               memcpy(&LeftBufferB[lp], &cacheBuff[lp*4], 2);
               memcpy(&RightBufferB[lp], &cacheBuff[2 + (lp*4)], 2);
          } 
     
          fillBufferA = TRUE;
     } 
     
     // return whether we ran out of data or not. FALSE = ran out, TRUE = more available.
     return moreData;
}


void Init_DAC_DMA( void )
{
     /* DMA Channel 0 set to DAC1RDAT */
     DMA0CONbits.AMODE = 0;                       /* Register Indirect with Post Increment */
     DMA0CONbits.MODE = 2;                        /* Continuous Mode with Ping-Pong Enabled */
     DMA0CONbits.DIR = 1;                         /* Ram-to-Peripheral Data Transfer */
     DMA0PAD = (volatile unsigned int)&DAC1RDAT;  /* Point DMA to DAC1RDAT */
     DMA0CNT = DMA_BUFF_SIZE-1;                   /* # DMA Requests */
     DMA0REQ = 78;                                /* Select DAC1RDAT as DMA Request Source */
     DMA0STA = __builtin_dmaoffset(RightBufferA);
     DMA0STB = __builtin_dmaoffset(RightBufferB);
     IFS0bits.DMA0IF = 0;                         /* Clear DMA Interrupt Flag */
     IEC0bits.DMA0IE = 1;                         /* Set DMA Interrupt Enable Bit */
     DMA0CONbits.CHEN = 1;                        /* Enable DMA Channel 0 */
     
     /* DMA Channel 1 set to DAC1LDAT */
     DMA1CONbits.AMODE = 0;                       /* Register Indirect with Post Increment */
     DMA1CONbits.MODE = 2;                        /* Continuous Mode with Ping-Pong Enabled */
     DMA1CONbits.DIR = 1;                         /* Ram-to-Peripheral Data Transfer */
     DMA1PAD = (volatile unsigned int)&DAC1LDAT;  /* Point DMA to DAC1LDAT */
     DMA1CNT = DMA_BUFF_SIZE-1;                   /* # DMA Requests */
     DMA1REQ = 79;                                /* Select DAC1LDAT as DMA Request Source */
     DMA1STA = __builtin_dmaoffset(LeftBufferA);
     DMA1STB = __builtin_dmaoffset(LeftBufferB);
     IFS0bits.DMA1IF = 0;                         /* Clear DMA Interrupt Flag */
     IEC0bits.DMA1IE = 1;                         /* Set DMA Interrupt Enable Bit */
     DMA1CONbits.CHEN = 1;                        /* Enable DMA Channel 1 */
     
     /* Setup ACLK for DAC */
     ACLKCON = 0x0780;        // Divide PLL clock by 1 for ACLK

     /* DAC1 Code */
     DAC1STATbits.ROEN = 1;   /* Right Channel DAC Output Enabled */ 
     DAC1STATbits.LOEN = 1;   /* Left Channel DAC Output Enabled */ 
     DAC1STATbits.RITYPE = 1; /* Right Channel Interrupt if FIFO is Empty */ 
     DAC1STATbits.LITYPE = 1; /* Left Channel Interrupt if FIFO is Empty */ 
     DAC1CONbits.AMPON = 0;   /* Amplifier is Disabled During Sleep/Idle Modes */
     DAC1CONbits.DACFDIV = 13;/* Divide Clock by 14 (value is 13) */
     DAC1CONbits.FORM = 1;    /* Data Format is 2's compliment signed */
     DAC1CONbits.DACEN = 1;   /* DAC1 Module Enabled */    
}



void __attribute__ ((interrupt, no_auto_psv)) _DMA0Interrupt(void)
{
     IFS0bits.DMA0IF = 0;     /* Clear DMA Channel 0 Interrupt Flag */
     rightEmpty = TRUE;
}

void __attribute__ ((interrupt, no_auto_psv)) _DMA1Interrupt(void)
{
     IFS0bits.DMA1IF = 0;     /* Clear DMA Channel 1 Interrupt Flag */
     leftEmpty = TRUE;
}

