;		FLEXTMR4.asm 	-- Jumper programmed Timer program
;		for PIC16F628A processor
;		written by Jim Rowe. Last revised 14/02/2011
;		(note that internal PIC clock is used, running at 4MHz
;		so one machine cycle (1mc) = 1us (+/-1%)
;		Timer's relay controlled via RA3 (1 = relay & LED on)
;
;	*******************************************************
 
	list	P=16F628A
	#include "p16f628A.inc"
	__config h'2131'	; intosc, pwrte enabled, RA5=MCLR etc
	
; 	storage of values from scanning jumper shunts
LK1jmp	equ		h'20'	; setting from LK1 (1-9 on time)
LK2jmp	equ		h'21'	; setting from LK2 (1-9 off time)
LK3jmp	equ		h'22'	; setting from LK3 (4/2/1 = h/m/s on)
LK4jmp	equ		h'23'	; setting from LK4 (1/0 = x10/x1 on)
LK5jmp	equ		h'24'	; setting from LK5 (4/2/1 = h/m/s off)
LK6jmp	equ		h'25'	; setting from LK6 (1/0 = x10/x1 off)

;	storage of timer values (on, off, current period)
Tons	equ		h'26'	; time on - seconds
Tonm	equ		h'27'	; time on - minutes
Tonh	equ		h'28'	; time on - hours

Toffs	equ		h'29'	; time off - seconds
Toffm	equ		h'2A'	; time off - minutes
Toffh	equ		h'2B'	; time off - hours

Tcsec	equ		h'2C'	; current time period - seconds
Tcmin	equ		h'2D'	; current time period - minutes
Tchr	equ		h'2E'	; current time period - hours
Tc50ms	equ		h'2F'	; current time period - 50ms R/T steps

; 	storage reg for timer mode, other flags
Flags	equ		h'30'	; b0 = mode flag (1 = 1-shot, 0=cont)
						; b1 = run flag (1 = timer running)
						; b2 = ON/OFF period flag (1 = on)
						; b3 = OVF flag (1 = OVF, 0 = none)
						
;	temporary storage registers
Temp	equ		h'31'	; working storage reg

; 	regs for saving context during interrupt servicing
WSave	equ		h'32'	; w reg stored here
SSave	equ		h'33'	; status reg stored here

; *********************************************************
;
; now program begins

	ORG		h'00'		; normal start vector
	GOTO Initialise
	ORG		h'04'		; interrupt servicing vector
	GOTO IntService
	
Initialise:
	BCF STATUS, RP0		; set for Bank0
	BCF STATUS, RP1
	MOVLW h'07'			; now disable comparators
	MOVWF CMCON
	CLRF INTCON			; disable all interrupts also
	CLRF Flags			; and clear the flags
	BSF STATUS, RP0		; now swing up to Bank1
	CLRF VRCON			; disable Vref module
	MOVLW h'FF'			; set RB0-7 as inputs
	MOVWF TRISB			; by loading config register
	MOVLW h'80'			; also set RA0-4 as outputs, but
	MOVWF TRISA			; RA7 as an input for sensing LK7
	MOVLW h'17'			; now enable PortB pullups, also
	MOVWF OPTION_REG	; set for TMR0, int clock & PS = 256
	BCF STATUS, RP0		; finally back down to Bank0
	BCF PORTA, 3		; and clear RA3 (turns off relay)
	CALL ScanJmprs		; now go scan the jumpers 
	CALL TmrSetup		; and set up timer from the settings

	BSF	Flags, 2		; we start with an ON prd, so set flag
	MOVF Tonh, 0		; Set up Tcurr values with ON prd vals
	MOVWF Tchr			; hours first
	MOVF Tonm, 0		; then minutes
	MOVWF Tcmin			; 
	MOVF Tons, 0		; and seconds
	MOVWF Tcsec			; 
	MOVLW h'14'			; set Tc50ms to 20d
	MOVWF Tc50ms
	MOVLW h'3D'			; preload Timer0 with 61d (also clears
	MOVWF TMR0			; prescaler, ready to start)
	BSF PORTA, 3		; then turn relay on,
	BSF Flags, 1		; set run flag to show we're running,
	BSF INTCON,5		; re-enable only T0IE interrupts and
	BSF INTCON,7		; finally set GIE to turn on interrupts

LoopStart:
	BTFSS Flags,1		; now check if run flag still set
	GOTO $+2			; no, so jump to deadloop
	GOTO LoopStart		; yes, so back to start of loop
	CLRF INTCON			; timer to stop, so interrupts off
	NOP					; and just twiddle thumbs
	GOTO $-1
;
;   Program's main loop ends here -- subroutines follow
;
;   *******************************************************
;
Chk4OFlow:
	; routine to check for mins or secs overflow, fix it by
	; subtracting 60 from number in W on entry. Remainder is
	; in W on exit & OVF flag set if remainder not = 0.
	; If remainder 0, OVF flag cleared & original content
	; of w restored before return
	MOVWF Temp			; first save W into Temp
	MOVLW h'3C'			; now place 60d/3Ch in W
	SUBWF Temp,0		; W = Temp - 60d
	BTFSS STATUS,Z		; now check if Z = 1 (zero remainder)
	GOTO $+3			; Z=0, so non zero remainder
	MOVF Temp,0			; z=1, so restore w contents
	GOTO $+5			; and clear OVF flag before exit			
	BTFSS STATUS,C		; Z=0 so remainder not 0; but OVF?
	GOTO $+3			; no, so just clear flag and go
	BSF Flags,3 		; yes OVF, so set OVF flag
	RETURN				; now leave (with remainder in w) 
	BCF Flags,3			; no OVF or 0 result, so clear flag
	RETURN				; before leaving 
;
;	*******************************************************
;
Multby10:
	; routine to multiply number in w reg (N) by 10d, then
	; return with result of 10*N in w reg
	MOVWF Temp			; save N in Temp
	BCF STATUS,C		; now clear carry bit
	RLF Temp,1			; rotate Temp L so it now has 2N
	MOVF Temp,0			; 2N is now in w reg also
	BCF STATUS,C		; clear carry bit again
	RLF Temp,1			; rotate Temp L so it now has 4N
	BCF STATUS,C		; clear carry bit again
	RLF Temp,1			; rotate Temp L so it now has 8N
	BCF STATUS,C		; clear carry bit again
	ADDWF Temp,0		; then add Temp to w, so w now has 10N
	RETURN				; then return with 10N in w
;
;	*******************************************************
;
ScanJmprs:
	;routine to scan jumpers LK1 - LK7 & set the timer
	BTFSS PORTA,7		; first check RA7 (mode link)
	GOTO $+3			; if not set, clear mode flag
	BSF Flags,0			; if  set, make mode flag = 1
	GOTO $+2			; and continue
	BCF Flags,0			
	MOVLW h'17'			; now pull up RA0-2, RA4
	MOVWF PORTA			; to prepare for scanning LK1-LK6

	BCF PORTA,4			; first pull down RA4 to find LK1
	BTFSC PORTB,0		; now check RB0 - is it 0?
	GOTO $+3			; if not, move up a notch
	MOVLW h'01'			; otherwise set w to 1
	GOTO Save1nOn		; and go save in LK1jmp, continue
	BTFSC PORTB,7		; now check RB7
	GOTO $+3			; if not 0, move up a notch
	MOVLW h'02'			; otherwise set w to 2
	GOTO Save1nOn		; and go save in LK1jmp, continue
	BTFSC PORTB,1		; now check RB1
	GOTO $+3			; if not 0, move up a notch
	MOVLW h'03'			; otherwise set w to 3
	GOTO Save1nOn		; and go save in LK1jmp, continue
	BTFSC PORTB,6		; now check RB6
	GOTO $+3			; if not 0, move up a notch
	MOVLW h'04'			; otherwise set w to 4
	GOTO Save1nOn		; and go save in LK1jmp, continue
	BTFSC PORTB,2		; now check RB2
	GOTO $+3			; if not 0, move up a notch
	MOVLW h'05'			; otherwise set w to 5
	GOTO Save1nOn		; and go save in LK1jmp, continue
	BTFSC PORTB,5		; now check RB5
	GOTO $+3			; if not 0, move up a notch
	MOVLW h'06'			; otherwise set w to 6
	GOTO Save1nOn		; and go save in LK1jmp, continue
	BTFSC PORTB,3		; now check RB3
	GOTO $+3			; if not 0, move up a notch
	MOVLW h'07'			; otherwise set w to 7
	GOTO Save1nOn		; and go save in LK1jmp, continue
	BTFSC PORTB,4		; now check RB4
	GOTO $+3			; if not 0, no jumper - so fix...
	MOVLW h'08'			; otherwise set w to 8
	GOTO Save1nOn		; and go save in LK1jmp, continue
	MOVLW h'09'			; no jumper, so set w to 9
Save1nOn:
	MOVWF LK1jmp		; save LK1 setting we found from w
						; now we look for LK2 jumper
	BSF PORTA,4			; so pull RA4 high again
	BCF PORTA,2			; and pull down RA2 instead
	BTFSC PORTB,0		; now check RB0
	GOTO $+3			; if not 0, move up a notch
	MOVLW h'01'			; otherwise set w to 1
	GOTO Save2nOn		; and go save in LK2jmp, continue
	BTFSC PORTB,7		; now check RB7
	GOTO $+3			; if not 0, move up a notch
	MOVLW h'02'			; otherwise set w to 2
	GOTO Save2nOn		; and go save in LK2jmp, continue
	BTFSC PORTB,1		; now check RB1
	GOTO $+3			; if not 0, move up a notch
	MOVLW h'03'			; otherwise set w to 3
	GOTO Save2nOn		; and go save in LK2jmp, continue
	BTFSC PORTB,6		; now check RB6
	GOTO $+3			; if not 0, move up a notch
	MOVLW h'04'			; otherwise set w to 4
	GOTO Save2nOn		; and go save in LK2jmp, continue
	BTFSC PORTB,2		; now check RB2
	GOTO $+3			; if not 0, move up a notch
	MOVLW h'05'			; otherwise set w to 5
	GOTO Save2nOn		; and go save in LK2jmp, continue
	BTFSC PORTB,5		; now check RB5
	GOTO $+3			; if not 0, move up a notch
	MOVLW h'06'			; otherwise set w to 6
	GOTO Save2nOn		; and go save in LK2jmp, continue
	BTFSC PORTB,3		; now check RB3
	GOTO $+3			; if not 0, move up a notch
	MOVLW h'07'			; otherwise set w to 7
	GOTO Save2nOn		; and go save in LK2jmp, continue
	BTFSC PORTB,4		; now check RB4
	GOTO $+3			; if not 0, move up a notch
	MOVLW h'08'			; otherwise set w to 8
	GOTO Save2nOn		; and go save in LK2jmp, continue
	MOVLW h'09'			; no jumper at all, so set w to 9
Save2nOn:
	MOVWF LK2jmp		; save LK2 setting we found from w
						; now we look for LK3, LK4...
	BSF PORTA,2			; now pull RA2 high again
	BCF PORTA,0			; and pull down RA0 instead
	BTFSC PORTB,4		; now check RB4 - clear?
	GOTO $+3			; if not 0, move down a notch
	MOVLW h'04'			; otherwise set w to 4 (= hours)
	GOTO Save3nOn		; and go save in LK3jmp, continue
	BTFSC PORTB,3		; now check RB3
	GOTO $+3			; if not 0, move down a notch
	MOVLW h'02'			; otherwise set w to 2 (= mins)
	GOTO Save3nOn		; and go save in LK3jmp, continue
	MOVLW h'01'			; must be seconds, so set w to 1
Save3nOn:
	MOVWF LK3jmp		; save w in LK3jmp
	BTFSC PORTB,2		; now check RB2 for LK4 jumper
	GOTO $+3			; if not 0, move on
	MOVLW h'01'			; otherwise set w to 1
	GOTO $+2
	MOVLW h'00'			; was down, so set w to 0
	MOVWF LK4jmp		; and save in LK4jmp
						; now we look for LK5 & 6...
	BSF PORTA,0			; so pull RA0 high again
	BCF PORTA,1			; and pull down RA1
	BTFSC PORTB,6		; now check RB6 - clear?
	GOTO $+3			; if not 0, move down a notch
	MOVLW h'04'			; otherwise set w to 4 (= hours)
	GOTO Save5nOn		; and go save in LK5jmp, continue
	BTFSC PORTB,1		; now check RB1
	GOTO $+3			; if not 0, move down a notch
	MOVLW h'02'			; otherwise set w to 2 (= mins)
	GOTO Save5nOn		; and go save in LK5jmp, continue
	MOVLW h'01'			; must be seconds, so set w to 1
Save5nOn:
	MOVWF LK5jmp		; save w in LK5jmp
	BTFSC PORTB,0		; now check RB0 for LK6 jumper
	GOTO $+3			; if not 0, move on
	MOVLW h'01'			; otherwise set w to 1
	GOTO $+2
	MOVLW h'00'			; otherwise set w to 0
	MOVWF LK6jmp		; and save in LK6jmp
	BSF PORTA,1			; we're done, so pull RA1 high again
	RETURN				; before leaving		
;
;	*******************************************************
;	
TmrSetup:
	; routine to set up timer according to jumper settings
	CLRF Tons			; first clear Ton settings
	CLRF Tonm
	CLRF Tonh
	CLRF Toffs			; and Toff settings
	CLRF Toffm
	CLRF Toffh
	CLRF Tcsec			; then clear current settings
	CLRF Tcmin
	CLRF Tchr
	BCF Flags,1			; finally clear run flag
	BCF Flags,2			; and ON/OFF flag
	BTFSS LK4jmp,0		; then check LK4 (1 = x10, 0 = x1)
	GOTO $+4			; if 0, just skip multiply step
	MOVF LK1jmp, 0		; but if 1, fetch LK1jmp into w
	CALL Multby10		; multiply it by 10
	MOVWF LK1jmp		; and resave it in LK1jmp
	BTFSS LK3jmp,2		; now is LK3 bit 2 set (= hours)?
	GOTO $+4			; if not, skip ahead
	MOVF LK1jmp,0		; if it is, fetch LK1jmp
	MOVWF Tonh			; and save in Tonh
	GOTO Offset			; then go to off time setting
	BTFSS LK3jmp,1		; how about LK3 bit 1 (set = mins)?
	GOTO $+9			; if not, skip ahead
	MOVF LK1jmp,0		; if it is, fetch LK1jmp
	MOVWF Tonm			; and save in Tonm
	CALL Chk4OFlow		; >=60? if so adjust Tonh & Tonm
	BTFSS Flags,3		; test OVF flag (1= OVF, 0 = none)
	GOTO Offset			; if none, skip to off time setting
	MOVWF Tonm			; if there was, save remainder to Tonm
	INCF Tonh,1			; and increment Tonh
	GOTO Offset			; then go to off time setting
	BTFSS LK3jmp,0		; how about LK3 bit 0 (set = secs)?
	GOTO Offset			; if not, skip to off time setting
	MOVF LK1jmp,0		; if it is, fetch LK1jmp in w
	MOVWF Tons			; and save in Tons
	CALL Chk4OFlow		; >=60? if so adjust Tonm & Tons
	BTFSS Flags,3		; test OVF flag (1 = OVF, 0 = none)
	GOTO Offset			; if none, skip to off time setting
	MOVWF Tons			; if there was, save remainder in Tons
	INCF Tonm, 1		; and increment Tonm
Offset:
	BTFSS LK6jmp,0		; then check LK6 (1 = x10, 0 = x1)
	GOTO $+4			; if 0, just skip multiply step
	MOVF LK2jmp, 0		; but if 1, fetch LK2jmp into w
	CALL Multby10		; multiply it by 10
	MOVWF LK2jmp		; and resave it in LK2jmp
	BTFSS LK5jmp,2		; now is LK5 bit 2 set ( = hours)?
	GOTO $+4			; if not, skip ahead
	MOVF LK2jmp,0		; if it is, fetch LK2jmp
	MOVWF Toffh			; and save in Toffh
	RETURN				; then leave already
	BTFSS LK5jmp,1		; how about LK5 bit 1 (set = mins)?
	GOTO $+9			; if not, skip ahead
	MOVF LK2jmp,0		; if it is, fetch LK2jmp
	MOVWF Toffm			; and save in Toffm
	CALL Chk4OFlow		; >=60? if so adjust Toffh & Toffm
	BTFSS Flags,3		; test OVF flag (1= OVF, 0 = none)
	RETURN				; if none, just leave
	MOVWF Toffm			; if there was, save remainder to Toffm
	INCF Toffh,1		; and increment Toffh
	RETURN				; then leave
	BTFSS LK5jmp,0		; how about LK5 bit 0 (set = secs)?
	RETURN				; if not, leave
	MOVF LK2jmp,0		; if it is, fetch LK2jmp in W
	MOVWF Toffs			; and save in Toffs
	CALL Chk4OFlow		; >=60? if so adjust Toffm & Toffs
	BTFSS Flags,3		; test OVF flag (1 = OVF, 0 = none)
	RETURN				; if none, just leave
	MOVWF Toffs			; if there was, save remainder in Toffs
	INCF Toffm, 1		; and increment Toffm	
	RETURN				; all done, so leave

; *********************************************************	
;	
; 	end of main routines -- interrupt routines follow
;	
IntService:
	; main routine to service interrupts from
	; Timer 0 (every 49.92ms when timer is going)
	MOVWF WSave			; first save context (w & status regs)
	SWAPF STATUS,0		; using SWAPF to avoid STATUS change
	MOVWF SSave
	
	DECFSZ Tc50ms,1		; decrement Tc50ms, skip if it ->0
	GOTO ResetTmr		; not =0, so reset Tmr0 & go

	MOVF Tcsec,1		; Tc50ms = 0, so check Tcsec
	BTFSC STATUS,Z		; skip if Z=0, because Tcsec not = 0
	GOTO $+3			; Tcsec = 0, so go check Tcmin
	DECFSZ Tcsec,1		; decrement Tcsec, skip if it now =0
	GOTO Reset50ms		; didn't so go, resetting Tc50ms

	MOVF Tcmin,1		; Tcsec did hit 0, so check Tcmin
	BTFSC STATUS,Z		; skip if Z=0, because Tcmin not 0
	GOTO $+3			; Tcmin already 0, so go check Tchr
	DECF Tcmin,1		; Tcmin not 0 yet, so decrement it
	GOTO ResetTcsec		; and go, resetting Tcsec, Tc50ms

	MOVF Tchr, 1		; Tcmin was 0, so now check Tchr
	BTFSC STATUS,Z		; skip if Z=0, because Tchr not 0
	GOTO $+3			; Z=1, so Tchr = 0. Time is up!
	DECF Tchr,1			; Tchr not 0, so decrement it
	GOTO ResetTcmin		; & go, resetting Tcmins,Tcsecs,Tc50ms

	BTFSC Flags,0		; current time is up, so test mode flag
	GOTO ClearRun		; = 1 (1 shot mode), so clear run flag
	BTFSC Flags, 2		; = 0 (cont mode), so test Ton/Toff flg
	GOTO ToffStart		; = 1, so turn rly off & begin OFF prd
	BSF	Flags, 2		; = 0, so begin an ON period. Set flag,
	BSF PORTA, 3		; turn relay on
	MOVF Tonh, 0		; & reset Tcurr settings with ON values
	MOVWF Tchr			; hours first
	MOVF Tonm, 0		; then minutes
	MOVWF Tcmin			; 
	MOVF Tons, 0		; and seconds
	MOVWF Tcsec			; 
	GOTO Reset50ms		; and finally reset Tc50ms & leave
ToffStart:
	BCF Flags, 2		; clear run flag
	BCF PORTA, 3		; turn relay off
	MOVF Toffh, 0		; & reset Tcurr settings with OFF values
	MOVWF Tchr			; hours first
	MOVF Toffm, 0		; then minutes
	MOVWF Tcmin			; 
	MOVF Toffs, 0		; and seconds
	MOVWF Tcsec			; 
	GOTO Reset50ms		; and finally reset Tc50ms & leave

ClearRun:
	BCF PORTA, 3		; it is 1-shot mode, so turn off relay
	BCF Flags, 1		; then clear RUN flag
	GOTO LeaveAtEnd		; and leave quietly
	
ResetTcmin:
	MOVLW h'3B'			; reset Tcmin to 59d
	MOVWF Tcmin
ResetTcsec:
	MOVLW h'3B'			; reset Tcsec to 59d
	MOVWF Tcsec
Reset50ms:
	MOVLW h'14'			; reset Tc50ms to 20d
	MOVWF Tc50ms	
ResetTmr:
	; routine to reset TMR0 for interrupts each 49.92ms,
	; restore context and return from interrupt
	MOVLW h'3D'			; preload Tmr0 with 61d, for right time
	MOVWF TMR0			; (also clears prescaler, ready to go)
	CLRF INTCON			; now clear all interrupt flags and
	BSF INTCON,5		; re-enable only T0IE interrupts
LeaveAtEnd:
	SWAPF SSave,0		; now restore context (status & w regs)
	MOVWF STATUS		;
	SWAPF WSave,1		; using SWAPFs to avoid changing STATUS
	SWAPF WSave,0		;
	RETFIE				; & return with TOS->PC, 1->GIE
	
;	*******************************************************
	END
	
