/****************************************************************************
*
*   Module:     videotest.c
*   Author:     Mike Hibbett
*   Date:       22/1/09
*   Purpose:    Example application using the PIC24 Video Library.
*               It implements a simple 1980's pong style game, with a twist.
*
****************************************************************************/

/**** INCLUDES *************************************************************/

#include "p24HJ128GP202.h"
#include <string.h>
#include <stdlib.h>

/* Include the PIC24 video library header file */
#include "pic24video.h"

/* logo.h contains a simple startup logo for the game */
#include "logo.h"


/**** APPLICATION CONFIG SETTINGS ******************************************/

 _FOSC( POSCMD_HS );
 _FOSCSEL( FNOSC_PRIPLL );
 _FWDT( FWDTEN_OFF & WINDIS_OFF );
 _FICD( ICS_PGD1 );


/**** CONSTANTS ************************************************************/
#define PLL_80MHZ       38

// constants that define the left and right goal area
#define BAR_TOP_START       8
#define BAR_TOP_END         ((VID_DEPTH_PIXELS/4)+8)
#define BAR_BOTTOM_START    (VID_DEPTH_PIXELS*3/4)
#define BAR_BOTTOM_END      (VID_DEPTH_PIXELS - 1)

/**** VARIABLES ************************************************************/
struct ball_t {
    int x,y;      // The coordinate of its top left corner
    unsigned char size;     // Width of the square ball, in pixels
    char x_shift;           // direction of travel. +ve right, -ve left
    char y_shift;           // direction of travel. +ve down, -ve up
} ball;

struct bat_t {
    int x,y;      // The coordinate of its top left corner
    unsigned char size;     // length of the bat, in pixels
} bat[2];

unsigned char leftScore, rightScore, finished;

/**** FORWARD DECLARATIONS *************************************************/



/**** CODE *****************************************************************/


/****************************************************************************
*
*   Function:   main
*   inputs:     none
*   returns:    exit code to OS environment
*   Purpose:    Main entry point of sample video application
*
****************************************************************************/

int  main(void)
{
        
    // Set processor to run at appropriate speed
    CLKDIVbits.PLLPOST = 0;
    PLLFBDbits.PLLDIV  = PLL_80MHZ;
    
    // Make digital pins I/O rather than analogue
    AD1PCFGL = 0xFFFF;    
    
    PIC24VideoInit();

    // Display a simple logo ( data converted from a .bmp file )
    memcpy(frameBuffer, logo, (VID_WIDTH_PIXELS/8) * VID_DEPTH_PIXELS);
    
    
    PIC24VideoDelays(5);
    
    do {
        PIC24VideoPutStr(38, 215, "Press middle button");
        PIC24VideoPutStr(38, 225, "     to start");  
        
        while ( PORTBbits.RB14 & PORTBbits.RB10 )
            ;

        PIC24VideoCls();

        PIC24VideoPutStr(50, 0, "Let's Play Tennis!");

        // draw playing area - top and bottom lines
        PIC24VideoDrawRect(0,8, VID_WIDTH_PIXELS-1, 8);
        PIC24VideoDrawRect(0,VID_DEPTH_PIXELS-1, VID_WIDTH_PIXELS-1, VID_DEPTH_PIXELS-1);
        
        // draw playing area - four side bars, with the 'goal' in the middle.
        // One goal on far left, the other on the far right.
        PIC24VideoDrawRect(0,BAR_TOP_START, 0, BAR_TOP_END); 
        PIC24VideoDrawRect(VID_WIDTH_PIXELS-1, BAR_TOP_START, VID_WIDTH_PIXELS-1, BAR_TOP_END);
        PIC24VideoDrawRect(0, BAR_BOTTOM_START, 0, BAR_BOTTOM_END);
        PIC24VideoDrawRect(VID_WIDTH_PIXELS-1, BAR_BOTTOM_START, VID_WIDTH_PIXELS-1, BAR_BOTTOM_END);
        
        // define the initial positions for the bats - in the centre of the goal
        bat[0].size = 18;
        bat[0].x = 2;
        bat[0].y = VID_DEPTH_PIXELS/2;
     
        bat[1].size = 18;
        bat[1].x = VID_WIDTH_PIXELS-3;
        bat[1].y = VID_DEPTH_PIXELS/2;
        
        // define the initial position of the ball
        ball.x = 10;
        ball.y = 10; 
        ball.size = 3; 
        ball.x_shift = 1;
        ball.y_shift = 1; // Move diagonally at a rate of 1 pixels every frame
        
        // draw initial position of bats
        PIC24VideoDrawRect(bat[0].x, bat[0].y, bat[0].x, bat[0].y + bat[0].size);
        PIC24VideoDrawRect(bat[1].x, bat[1].y, bat[1].x, bat[1].y + bat[1].size);

        finished = 0;
        
        leftScore=0;
        rightScore=0;
        
        do {
            
            // wait for tick/frame update
            PIC24VideoWaitFrame();
            
            
            // erase current objects
            PIC24VideoClearRect(ball.x, ball.y, ball.x+ball.size, ball.y+ball.size);
            
            
            // update current objects positions
            ball.x += ball.x_shift;
            ball.y += ball.y_shift;
                
               
            // test for ball boundary conditions - wall. Change directions
            if ( ball.y <= 9 ) {
                ball.y = 9;
                ball.y_shift *= -1;
                PIC24VideoPlayNote(60000, 4);
            }
    
            if ( (ball.y+ball.size) >= (VID_DEPTH_PIXELS-2) ) {
                ball.y = VID_DEPTH_PIXELS-2 - ball.size;
                ball.y_shift *= -1;
                PIC24VideoPlayNote(60000, 4);
            }
                        
            if ( ball.x <= 1 ) {
                ball.x = 1;
                ball.x_shift *= -1;
                PIC24VideoPlayNote(60000, 4);
            }
            
            if ( (ball.x+ball.size) >= (VID_WIDTH_PIXELS-2) ) {
                ball.x = VID_WIDTH_PIXELS-2 - ball.size;
                ball.x_shift *= -1;
                PIC24VideoPlayNote(60000, 4);
            }
            
            // test for ball boundary conditions - goal, left side
            if ( (ball.x == 1) && ( (ball.y > BAR_TOP_END) && (ball.y < BAR_BOTTOM_START))) {
                // score a point. reset ball position.
                PIC24VideoPlayNote(40000, 16);
                rightScore++;
                PIC24VideoPutChar(VID_WIDTH_PIXELS-6, 0, '0' + rightScore);
    
                ball.x = 10;
                ball.y = (TMR1 & 0x3f) + 10;    // Start of in a random location
                ball.x_shift = 1;
                ball.y_shift = 1; // Move diagonally at a rate of 1 pixels every frame

                PIC24VideoDelays(1);
            }
    
            // test for ball boundary conditions - goal, left side
            if ( ((ball.x+ball.size) == (VID_WIDTH_PIXELS-2)) && ( (ball.y > BAR_TOP_END) && (ball.y < BAR_BOTTOM_START))) {
                // score a point. reset ball position.
                PIC24VideoPlayNote(40000, 16);
                leftScore++;
                PIC24VideoPutChar(0, 0, '0' + leftScore);
    
                ball.x = VID_WIDTH_PIXELS-11;
                ball.y = (TMR1 & 0x3f) + 10;    // Start of in a random location
                ball.x_shift = -1;
                ball.y_shift = 1; // Move diagonally at a rate of 1 pixels every frame

                PIC24VideoDelays(1);
            }
            
            
            // test for ball boundary conditions - right bat
            if ((ball.x+ball.size) == (VID_WIDTH_PIXELS-4)) {
                char diff = ball.y-bat[1].y;
                
                switch (diff) {
                    case 0: 
                    case 1: 
                    case 2: if (PORTBbits.RB14 == 0) ball.x_shift = -2; else ball.x_shift *= -1;
                             ball.y_shift = -3; // was -1
                             PIC24VideoPlayNote(30000, 3); break;
                    case 3: 
                    case 4: if (PORTBbits.RB14 == 0) ball.x_shift = -2; else ball.x_shift *= -1;
                             ball.y_shift = -2; // was -1
                             PIC24VideoPlayNote(30000, 3); break;
                    case 5: 
                    case 6:  if (PORTBbits.RB14 == 0) ball.x_shift = -2; else ball.x_shift *= -1;
                             ball.y_shift = -1; // was -1
                             PIC24VideoPlayNote(30000, 3); break;
                    case 7:
                    case 8:
                    case 9: 
                    case 10: 
                    case 11: 
                            if (PORTBbits.RB14 == 0) ball.x_shift = -2; else ball.x_shift *= -1;
                             ball.y_shift = 0;  
                             PIC24VideoPlayNote(30000, 3); break;
                    case 12: 
                    case 13: if (PORTBbits.RB14 == 0) ball.x_shift = -2; else ball.x_shift *= -1;
                             ball.y_shift = 1;
                             PIC24VideoPlayNote(30000, 3); break;
                    case 14: 
                    case 15: if (PORTBbits.RB14 == 0) ball.x_shift = -2; else ball.x_shift *= -1;
                             ball.y_shift = 2;
                             PIC24VideoPlayNote(30000, 3); break;
                    case 16: 
                    case 17: 
                    case 18: if (PORTBbits.RB14 == 0) ball.x_shift = -2; else ball.x_shift *= -1;
                             ball.y_shift = 3;  // was 1
                             PIC24VideoPlayNote(30000, 3); break;
                }
            }
    
            // test for ball boundary conditions - left bat
            if (ball.x == 3) {
                char diff = ball.y-bat[0].y;
                
                switch (diff) {
                    case 0: 
                    case 1: 
                    case 2:  if (PORTBbits.RB10 == 0) ball.x_shift = 2; else ball.x_shift *= -1;
                             ball.y_shift = -3;  
                             PIC24VideoPlayNote(30000, 3); 
                             break;
                    case 3: 
                    case 4:  if (PORTBbits.RB10 == 0) ball.x_shift = 2; else ball.x_shift *= -1;
                             ball.y_shift = -2;  
                             PIC24VideoPlayNote(30000, 3); 
                             break;
                    case 5: 
                    case 6:  if (PORTBbits.RB10 == 0) ball.x_shift = 2; else ball.x_shift *= -1;
                             ball.y_shift = -1;  
                             PIC24VideoPlayNote(30000, 3); 
                             break;
                    case 7:
                    case 8: 
                    case 9: 
                    case 10: 
                    case 11: if (PORTBbits.RB10 == 0) ball.x_shift = 2; else ball.x_shift *= -1;
                             ball.y_shift = 0;  
                             PIC24VideoPlayNote(30000, 3); 
                             break;
                    case 12: 
                    case 13: if (PORTBbits.RB10 == 0) ball.x_shift = 2; else ball.x_shift *= -1;
                             ball.y_shift = 1;  
                             PIC24VideoPlayNote(30000, 3); 
                             break;
                    case 14: 
                    case 15: if (PORTBbits.RB10 == 0) ball.x_shift = 2; else ball.x_shift *= -1;
                             ball.y_shift = 2;  
                             PIC24VideoPlayNote(30000, 3); 
                             break;
                    case 16: 
                    case 17: 
                    case 18: if (PORTBbits.RB10 == 0) ball.x_shift = 2; else ball.x_shift *= -1;
                             ball.y_shift = 3;  
                             PIC24VideoPlayNote(30000, 3); 
                             break;
                }
            }
            
            
            // Test for bat movement
            if ( PORTBbits.RB7 == 0 ) {
                if ( bat[0].y > 10) {
                    PIC24VideoClearRect(bat[0].x, bat[0].y, bat[0].x, bat[0].y + bat[0].size);
                    bat[0].y -= 2;
                    PIC24VideoDrawRect(bat[0].x, bat[0].y, bat[0].x, bat[0].y + bat[0].size);
                }
            }
            
            if ( PORTBbits.RB12 == 0 ) {
                if ( (bat[0].y + bat[0].size) < (VID_DEPTH_PIXELS - 2)) {
                    PIC24VideoClearRect(bat[0].x, bat[0].y, bat[0].x, bat[0].y + bat[0].size);
                    bat[0].y += 2;
                    PIC24VideoDrawRect(bat[0].x, bat[0].y, bat[0].x, bat[0].y + bat[0].size);
                }
            }
            
            if ( PORTBbits.RB13 == 0 ) {
                if ( bat[1].y > 10) {
                    PIC24VideoClearRect(bat[1].x, bat[1].y, bat[1].x, bat[1].y + bat[1].size);
                    bat[1].y -= 2;
                    PIC24VideoDrawRect(bat[1].x, bat[1].y, bat[1].x, bat[1].y + bat[1].size);
                }
            }
            
            if ( PORTBbits.RB15 == 0 ) {
                if ( (bat[1].y + bat[1].size) < (VID_DEPTH_PIXELS - 2)) {
                    PIC24VideoClearRect(bat[1].x, bat[1].y, bat[1].x, bat[1].y + bat[1].size);
                    bat[1].y += 2;
                    PIC24VideoDrawRect(bat[1].x, bat[1].y, bat[1].x, bat[1].y + bat[1].size);
                }
            }
            
            // re-draw ball
            PIC24VideoDrawRect(ball.x, ball.y, ball.x+ball.size, ball.y+ball.size);
            
            
            // Check for game over 
            if ( rightScore == 9 ) {
                PIC24VideoPutStr(VID_WIDTH_PIXELS-42, 0, "Winner!");  
                finished = 1;
            }
    
            if ( leftScore == 9 ) {
                PIC24VideoPutStr(0, 0, "Winner!");  
                finished = 1;
            }
      
        } while ( !finished ); 
            
    } while (1); // loop forever
    
    return 0;
}
