/*****************************************************************************        
*
*   Module:     menu.c
*               
*   Author:     Mike Hibbett 
*                                                                  
*   Version:    0.1 29/10/07                                                  
*
*               Demonstrates a user interface based on a state machine
*
*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>

#include <xlcd.h>
#include <delays.h>

/* Set the configuration registers as we want them */
#pragma config WDT=OFF, OSC=INTIO7, LVP=OFF
#pragma config MCLRE = OFF, PBADEN = OFF

/* Useful names for constants used in the program */
#define ADC_OFF_VAL 0x0F


/* Names for the events that can be handled by the system */
/* An enum would be better here, but defines are simpler */
#define NUMEVENTS   6
#define STARTUP 	0
#define RIGHTKEY 	1
#define LEFTKEY		2
#define UPKEY		3
#define DOWNKEY		4
#define ACTIONKEY   5


/* Names for the states that the state machine can be in. */
/* Again, an enum would be better here, but defines are simpler */
#define NUMSTATES	   7
#define STARTUP_STATE  0
#define INITLCD_STATE  1
#define DSPTEMP_STATE  2
#define HIGHTEMP_STATE 3
#define LOWTEMP_STATE  4
#define EDITHIGH_STATE 5
#define EDITLOW_STATE  6


/* Forward declarations, as required by 'C' */
void initHardware( void );
unsigned char getKey( void );
void UIWait10ms(unsigned char );
unsigned char readCurrentTemp( void );
void heaterOn( void );
void heaterOff( void );

void menuState( char );

void dspTempValue( unsigned char );

/* List of different state handling routines */
char sInitLCD( void );
char sDspTemp( void );
char sDspHighTemp( void );
char sDspLowTemp( void ); 
char sEditHigh( void );
char sEditLow( void );
char sIncHigh( void );
char sDecHigh( void );
char sIncLow( void );
char sDecLow( void );

/* define a table of function pointers */
/*   STARTUP,RIGHTKEY,LEFTKEY,UPKEY,DOWNKEY,ACTIONKEY */

char (*states[NUMSTATES][NUMEVENTS])(void) = {
	{ sInitLCD, NULL, NULL, NULL, NULL, NULL },  /* STARTUP_STATE */
	{ NULL, sDspTemp, NULL, NULL, NULL, NULL }, /* INITLCD_STATE */
	{ NULL, sDspHighTemp, sInitLCD, NULL, NULL, NULL }, /* DSPTEMP_STATE */
	{ NULL, sDspLowTemp, sDspTemp, NULL, NULL, sEditHigh }, /* HIGHTEMP_STATE */
	{ NULL, NULL, sDspHighTemp, NULL, NULL, sEditLow }, /* LOWTEMP_STATE */
	{ NULL, NULL, NULL, sIncHigh, sDecHigh, sDspHighTemp }, /* EDITHIGH_STATE */
	{ NULL, NULL, NULL, sIncLow, sDecLow, sDspLowTemp }  /* EDITLOW_STATE */
};
	

unsigned char currentTemp = 20;
unsigned char highTemp = 30;
unsigned char lowTemp = 10;

/*****************************************************************************
*
*   Function : main
*              The entry point to our C program
*
*   Input:     None.
*
*****************************************************************************/
void main(void)
{
	unsigned char key;
	
	initHardware();
	
	menuState(STARTUP);
	
	/* loop round applictaion, forever */
	do {
		key = getKey();
		
		if ( key )
			menuState(key);
		
		/* Get current heater setting */
		currentTemp = readCurrentTemp();
		
		/* If too hot or cold, turn the heater off/on */
		if ( currentTemp <= lowTemp ) 
			heaterOn();

		if ( currentTemp >= highTemp ) 
			heaterOff();
				
	} while (1);
	
}



/*****************************************************************************
*
*   Function : menuState
*              The state machine code. This uses a 'static' variable
*              to remember what the current state is, so that incoming events
*              can be handled properly.
*
*   Input:     None.
*
*****************************************************************************/
void menuState( char eventType )
{
	static char currentState = STARTUP_STATE;
	
	char newState= -1;
	
	if ( states[currentState][eventType] != NULL ) 
		newState = (*states[currentState][eventType])();
	
	if (newState >= 0)
		currentState = newState;
}


/*****************************************************************************
*
*   Function : various state change routines
*              The various functions, referenced in the array variable states,
*              that get called by menuState.
*
*              Notice how simple the logic in each of these is; the complexity 
*              (of the user navigation) is handled by the 'states' array variable
*              
*   Input:     None 
*
*	Output:    New state value, or -1 for no state change.
*
*****************************************************************************/

char sInitLCD( void )
{
	while( BusyXLCD() );
	SetDDRamAddr( 0 );
	putrsXLCD( "TEMPERATURE    " );
	while( BusyXLCD() );
	SetDDRamAddr( 40 );
	putrsXLCD( "CONTROLLER     " );
	
	return INITLCD_STATE;
}


char sDspTemp( void )
{
	while( BusyXLCD() );
	SetDDRamAddr( 0 );
	putrsXLCD( "CURRENT TEMP IS " );
	while( BusyXLCD() );
	SetDDRamAddr( 40 );
	
	dspTempValue(currentTemp);
	
	return DSPTEMP_STATE;
}

char sDspHighTemp( void )
{
	while( BusyXLCD() );
	SetDDRamAddr( 0 );
	putrsXLCD( "HIGH TEMP IS    " );
	while( BusyXLCD() );
	SetDDRamAddr( 40 );
	
	dspTempValue(highTemp);

	return HIGHTEMP_STATE;
}

char sDspLowTemp( void )
{
	while( BusyXLCD() );
	SetDDRamAddr( 0 );
	putrsXLCD( "LOW TEMP IS     " );
	while( BusyXLCD() );
	SetDDRamAddr( 40 );
	
	dspTempValue(lowTemp);

	return LOWTEMP_STATE;
}
 
char sEditHigh( void )
{
	while( BusyXLCD() );
	SetDDRamAddr( 0 );
	putrsXLCD( "EDIT HIGH TEMP: " );
	while( BusyXLCD() );
	SetDDRamAddr( 40 );
	
	dspTempValue(highTemp);

	return EDITHIGH_STATE;             
}
 
char sEditLow( void )
{
	while( BusyXLCD() );
	SetDDRamAddr( 0 );
	putrsXLCD( "EDIT LOW TEMP:  " );
	while( BusyXLCD() );
	SetDDRamAddr( 40 );
	
	dspTempValue(lowTemp);

	return EDITLOW_STATE;             
}
 

char sIncHigh( void )
{
	if (highTemp < 99)
		highTemp = highTemp + 1;
	
	while( BusyXLCD() );
	SetDDRamAddr( 40 );
	
	dspTempValue(highTemp);
	
	return -1; /* No change of state */
}
 

char sDecHigh( void )
{
	if (highTemp > 0)
		highTemp = highTemp - 1;
	
	while( BusyXLCD() );
	SetDDRamAddr( 40 );
	
	dspTempValue(highTemp);

	return -1; /* No change of state */
}
 

char sIncLow( void )
{
	if (lowTemp < 99)
		lowTemp = lowTemp + 1;
	
	while( BusyXLCD() );
	SetDDRamAddr( 40 );
	
	dspTempValue(lowTemp);
	
	return -1; /* No change of state */
}
 

char sDecLow( void )
{
	if (lowTemp > 0)
		lowTemp = lowTemp - 1;
	
	while( BusyXLCD() );
	SetDDRamAddr( 40 );
	
	dspTempValue(lowTemp);

	return -1; /* No change of state */
}
 
 

/*****************************************************************************
*
*   Function : dspTempValue
*              A helper function that displays a temperature, in the range
*              0 - 99 degrees, on the second line of the LCD.
*              The LCD should already be set to the start of the second line
*
*   Input:     value to be displayed.
*
*****************************************************************************/
void dspTempValue( unsigned char tempVal )
{
	while( BusyXLCD() );

	if ( tempVal < 10 ) {
		putcXLCD(' ');
	} else {
		putcXLCD('0' + (tempVal / 10));
	}
	
	while( BusyXLCD() );
	putcXLCD('0' + (tempVal % 10));
	
	putrsXLCD( " Degrees C    " );
}


/*****************************************************************************
*
*   Function : initHardware
*              Configures the speed of the on-board oscillator,
*              sets up the I/O ports and initialises the LCD
*
*   Input:     None.
*
*****************************************************************************/
void initHardware()
{
	/* Set oscillator to 8MHz. */
	OSCCON = 0x73; 

    /* Allow PORTB pins to be set to digital I/O */
    ADCON1 = ADC_OFF_VAL;

	/* Make PORTB output, for driving LCD 
	PORTB = 0xFF;
	TRISB = 0;*/
	
	/* First, delay for 1/2s for things to settle */
	UIWait10ms(50);
	
	/* Make port RC inputs, for the keys */
	TRISC=0xFF;
		
	/* Initialise the LCD */
	OpenXLCD( FOUR_BIT & LINES_5X7 );
	while( BusyXLCD() );
	WriteCmdXLCD(BLINK_OFF & CURSOR_OFF);
	
}



/*****************************************************************************
*
*   Function : getKey
*              Returns the status of the keyboard: 0 for no key,
*              or the value of the key pressed.
*              The binary value for the keypress matches the value for the
*              'Key event' event values; this simplified the translation
*              of a key press into a key event - there is none!
*
*   Input:     None
*   Output:    0 or key event value.
*
*****************************************************************************/
unsigned char getKey( void )
{
	unsigned char keyValue;

	/* has a key (possibly) been pressed? */
	/* If not, just return */
	
	keyValue = PORTC & 0x1F;
	
	if (keyValue != 0x1F) {

		do {	
			/* Wait for the key to be released */	
	
			UIWait10ms(2);
	
			do {
				; /* nothing */
			} while ( (PORTC & 0x1F) != 0x1F );
			
			/* check again */
			UIWait10ms(2);
		} while ((PORTC & 0x1F) != 0x1F);
			
	}

	/* return the appropriate key event value */	
	switch (keyValue) {
		case 0x1E: return RIGHTKEY;
		case 0x1D: return LEFTKEY;
		case 0x1B: return UPKEY;
		case 0x17: return DOWNKEY;
		case 0x0F: return ACTIONKEY;
		default: return 0;
	}
	
}



/*****************************************************************************
*
*   Function : UIWait10ms
*              delays for a multiple of 10ms
*              sets up the I/O ports and initialises the LCD
*
*   Input:     multiple of 10ms, 1 .. 255
*
*****************************************************************************/
void UIWait10ms(unsigned char num)
{
	do {
		DelayXLCD();
		DelayXLCD();		
	} while (--num);
	
}



/*****************************************************************
*
*  The routines below are needed for the C-Library's LCD display
*  routines to work. The effectively implement the 'target' 
*  specific requirements, such as delay routines. The C-library
*  does not know how fast your processor is running, for example!
*
*****************************************************************/


void DelayFor18TCY(void)
{
	// Delay for 18 clock cycles
	_asm
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	_endasm
}


void DelayXLCD(void)
{
	// delay for 5 ms	
    // Cycles = (TimeDelay * Fosc) / 4
    // Cycles = (5ms * 8MHz) / 4
    // Cycles = 100,000}
	Delay1KTCYx(10); 
}

void DelayPORXLCD(void)
{
	// Delay for 15ms
	DelayXLCD();
	DelayXLCD();
	DelayXLCD();
}


/* Dummy functions follow */
unsigned char readCurrentTemp( void )
{
	return 22;
}

void heaterOn( void )
{
}

void heaterOff( void )
{
}
