;*****************************************************************************        
;
;   Module:     spi.inc
;               
;   Author:     Mike Hibbett  mike.hibbett@gmail.com
;                                                                  
;   Version:    0.1 21/11/05                                            
;
;               Functions to communicate with an SPI EEPROM or FLASH
;
;*****************************************************************************        




; Use Equates directives to give constants meaningful names


; Do not change these two definitions
SPI_USE_HW_MODULE       EQU 0
SPI_USE_BIT_BASHED      EQU 1


; ***** NOTE ******
; This definition is used to determine if the SPI interface 
; uses the inbuilt module or is bit bashed. Select one of
; the SPI_USE defintions above.

SPI_MODULE_MODE         EQU SPI_USE_HW_MODULE




;*****************************************************************************        
;
; SPI Bus hardware constants
;
;*****************************************************************************        
SSPSTAT_VAL             EQU 0x00    ; 00000000
SSPCON1_VAL             EQU 0x32    ; 00110010

PORTC_APP_TRIS_VAL      EQU 0x90    ; 10010000
DEVICE_CS_BIT           EQU 2       ; The bit number, used in port bit operations
DEVICE_SCK_BIT          EQU 3       ; The bit number, used in port bit operations
DEVICE_SI_BIT           EQU 4       ; The bit number, used in port bit operations
DEVICE_SO_BIT           EQU 5       ; The bit number, used in port bit operations



;*****************************************************************************        
;
; SPI EEPROM/FLASH constants
;
;*****************************************************************************        
DEVICE_WREN             EQU 0x06    ; Write enable
DEVICE_WRDI             EQU 0x04    ; Write disable
DEVICE_RDSR             EQU 0x05    ; Read status register  
DEVICE_WRSR             EQU 0x01    ; Write status register
DEVICE_READ             EQU 0x03    ; Read data from memory
DEVICE_RDID             EQU 0x9f    ; Read ID Codes from device
DEVICE_SECTOR_ERASE     EQU 0xD8    ; Sector Erase command
DEVICE_BULK_ERASE       EQU 0xC7    ; Bulk Erase command
DEVICE_WRITE            EQU 0x02    ; Write data to memory



;*****************************************************************************        
;
; RAM Variables
;
;*****************************************************************************        
bbTX                    EQU 0x11    ; Transmit shift register for bit-bashing output
bbRX                    EQU 0x12    ; Receive shift register for bit-bashing input
bbCount                 EQU 0x13    ; Counts the 8 clocks for bit bashing




;*****************************************************************************        
;
;   Macro :  HBIT_DELAY
;            Helper macro that pauses for ONE HALF of an SPI bit period
;            This is used by the bit-bashing implementation of SPI
;            As SPI is a synchronous protocol you can decide on your own clock
;            speed, up to the limit of the device. This version runs at about 
;            250KHz on a 4MHz CPU clock.
;
;   Input:   None
;
;   Output:  None
;
;*****************************************************************************        
HBIT_DELAY  macro  
            nop
            nop
            endm
            
            

;*****************************************************************************        
;
;   Function:  TXRX_SPI
;               Performs the low level SPI functionality;
;               Transmitting a byte and simultaneously receiving a byte
;   
;               SPI is a funny protocol. As it has dedicated transmit and 
;               receive lines, it can transmit a byte and receive one at the
;               same time. But few devices use this facility.
;               So when you are sending a byte, rubbish is received.
;               And when you want to receive a byte, you actually have to transmit
;               a byte, even though it is unused. Thats just the way it works!
;
;               This call blocks while it waits for the transfer to complete
;               As it uses the built in SPI module, you could set the CPU
;               up to interrupt when done, enabling you to do other stuff
;               rather than just wait. Blocking simplifies the example.
;
;               This function will implement either Bit-Bashed or SPI Module
;               communications, depending on the setting of the SPI_MODULE_MODE
;               symbol at the top of the file
;               
;
;   Input:      Value to transmit in W
;   
;   Output:     Value received in W
;
;*****************************************************************************        
TXRX_SPI 
    IF (SPI_MODULE_MODE == SPI_USE_HW_MODULE) 
    
    ; Use the HW module to send and receive the data 
    movwf   SSPBUF              ; send data
    btfss   SSPSTAT, BF         ; wait for transfer to complete
    bra     $-1
    movf    SSPBUF, W           ; Get data
    
    ELSE
    
    ; Use bit bashing to get the byte out and data in
    movwf   bbTX
    clrf    bbRX
    movlw   8
    movwf   bbCount
    
SPILoop    
    ; Transmit a databit, most significant first
    rlcf    bbTX,F
    btfss   STATUS, C
    bcf     LATC, DEVICE_SO_BIT
    btfsc   STATUS, C
    bsf     LATC, DEVICE_SO_BIT
    
    bcf     LATC, DEVICE_SCK_BIT ; Assert the clock ( drive to 0 )
    
    HBIT_DELAY                   ; delay for a 1/2 clock period
    
    ; Now sample the input signal, and store in bbRX
    rlcf    bbRX,F
    btfss   PORTC, DEVICE_SI_BIT
    bcf     bbRX, 0
    btfsc   PORTC, DEVICE_SI_BIT
    bsf     bbRX, 0

    bsf     LATC, DEVICE_SCK_BIT ; De-assert the clock ( drive to 1 )
        
    HBIT_DELAY                   ; delay for a 1/2 clock period

    decfsz  bbCount,F
    bra     SPILoop
    movf    bbRX, W
    
    ENDIF
    
    return




;*****************************************************************************        
;
;   Function :  SPIHWInit
;               configures the SPI peripherial hardware into mode 1,1
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
SPIHWInit

    ; Configure the Chip Select pin as an output, and set it to disabled (high)
    movlw   PORTC_APP_TRIS_VAL
    movwf   TRISC 
    bsf     LATC, DEVICE_CS_BIT

    IF (SPI_MODULE_MODE == SPI_USE_HW_MODULE) 

    ; Setup the HW module to send and receive the data
    movlw   SSPSTAT_VAL
    movwf   SSPSTAT
    
    movlw   SSPCON1_VAL
    movwf   SSPCON1
   
    ELSE
    
    ; We are using bit-bashing, so setup the IO pins and disable the SPI Module

    bcf     SSPCON1, SSPEN      ; Disable SPI module

    bsf     LATC, DEVICE_SCK_BIT
        
    ENDIF
    
    return



