
;*****************************************************************************        
;
;   Module:     app.inc
;               
;   Author:     Mike Hibbett, mike.hibbett@gmail.com 
;                                                                  
;   Version:    1.0 26/07/06                                              
;
;               The main user application code , entered
;               following execution of the bootloader. 
;
;               All your application code should go in here, or be #included.
;               The exception is interrupt routines, which should go in
;               interrupts.inc and the setting of the config registers, which 
;               should go in config.inc
;
;               CS is on RC2; SPI stuff on normal pins; RS232 on normal pins
;
;*****************************************************************************        


;*****************************************************************************        
;
; MACROS
;
;*****************************************************************************        

DISPLAY_STR     macro   str_addr
                clrf    TBLPTRU     ; Memory is limited to 64KB
                movlw   HIGH str_addr 
                movwf   TBLPTRH
                movlw   LOW str_addr 
                movwf   TBLPTRL
                call    DspRomStr                
                endm


    ; Pull in the source code for the SPI low level routines
    #include "spi.inc"    

    ; Pull in the source code for the MMC Card low level routines
    #include "mmc.inc"    

    ; Pull in the source code for the FAT16 driver
    #include "fat16.inc"    



;*****************************************************************************        
;
; RAM Variables
;
;*****************************************************************************        
tmpHexByte  EQU 0x14        ; used by DspHexByte for formatting text
MMCCmd      EQU 0x15
MMCOp1      EQU 0x16
MMCOp2      EQU 0x17
MMCOp3      EQU 0x18
MMCOp4      EQU 0x19
MMCOp5      EQU 0x1A
tmpCount    EQU 0x1B        ; General purpose counter
blockSize   EQU 0x1C        ; Block size indicator; 0 = one, 1 = 512
row         EQU 0x1D        ; Used to display 512 byte block buffer
column      EQU 0x1E        ; Used to display 512 byte block buffer
indH        EQU 0x1F        ; high byte of index into buffer
indL        EQU 0x20        ; low byte of index into buffer
tmpByte     EQU 0x21
codePtrL    EQU 0x22
codePtrH    EQU 0x23
cmdPtrH     EQU 0x24
cmdPtrL     EQU 0x25
index       EQU 0x26        ; temporary index counter
cmdLine     EQU 0x60        ; 32 byte command line buffer ( shared with FAT directory data space )


; FAT interface variables

FAT_Filename    EQU 0x27        ; 11 bytes long.
FAT_flags1      EQU 0x32        ; 1 byte.
FAT_Part_Add    EQU 0x33        ; 4 bytes. Start address of partition
FAT_FAT_Add     EQU 0x37        ; 4 bytes. Start address of first FAT
FAT_Dir_Add     EQU 0x3B        ; 4 bytes. Start of directory table
FAT_Data_Add    EQU 0x3F        ; 4 bytes. Start of Data.
free1           EQU 0x43        ; 1 byte - number of FAT tables
FAT_SecInClust  EQU 0x44        ; 1 byte - current sector in a cluster
FAT_FileC       EQU 0x45        ; 2 bytes - current cluster
FAT_FAT_Size    EQU 0x47        ; 4 bytes - size of fat table, in bytes
FAT_File_Size   EQU 0x4B        ; 4 bytes - size of current, open file
FAT_File_Ind    EQU 0x4F        ; 4 bytes - byte addr of a sector in file : Current for read
FAT_tmp_count   EQU 0x53        ; 2 byte - used by open file to find directory entry
FAT_Dir_Sec     EQU 0x55        ; 1 byte - used during directory listings
FAT_Dir_Ind     EQU 0x56        ; 1 byte - used during directory listings 
FAT_Dir_Pos     EQU 0x57        ; 4 bytes - byte index of current directory sector
FAT_File_Pos    EQU 0x5B        ; 4 bytes - Start of current sector on an open file, in bytes
FAT_ClusterS    EQU 0x5F        ; 1 byte - number of sectors in the cluster
FAT_Dir_Entry   EQU 0x60        ; 32 bytes, a directory entry. YES, we re-use cmdLine
FAT_Tmp_Long    EQU 0x80        ; 4 bytes.


; We use 512 bytes from this address
; This will be a buffer for a 'sector' from the file system on the media card.
BLOCK_BUFFER_ADDRESS  EQU 0x100  


; Bit fields in various flags
flags1_FAT_INIT_ERROR       EQU     0   ; Typically, no file system found
flags1_FAT_INIT_COMPLETE    EQU     1   ; Indicates that initialisation has finished
flags1_FAT_NO_MBR           EQU     2   ; Indicates that there is no MBR present
flags1_FAT_FILE_IS_OPEN     EQU     3   ; Indicates that the user has a file open



;*****************************************************************************        
;
;   Function :  Main
;               This is the entry point of your actual application program
;
;               This code demonstrates the FAT16 driver software
;
;   Input:      None.
;
;   Output:     None - it never returns
;
;*****************************************************************************        
Main
    ; Do the basic hardware initialisation
    call    AppHWInit
    
    ; Configure the PICs internal SPI module for communication
    call    SPIHWInit
    
    ; Show the welcome screen, and help display.
    DISPLAY_STR STR_CRLF
    DISPLAY_STR STR_MAIN
    DISPLAY_STR STR_CRLFPROMPT
    call    CmdHelpScreen
    
    ; Reset MMC card following power up
    call    MMCReset

    ; Software reset of card
    call    MMCInit
    
    ; Init the FAT16 driver
    call    FAT16Init    
    
    DISPLAY_STR STR_CRLFPROMPT

    ; Test the flasg flags1_FAT_INIT_ERROR and flags1_FAT_NO_MBR
    ; and report over the rs232 link; This lets the user know if the
    ; FAT16 driver found a file system on the media
    
    btfss   FAT_flags1, flags1_FAT_INIT_ERROR    
    goto    finit_ok

    DISPLAY_STR STR_FAT_NO

finit_ok
    DISPLAY_STR STR_FAT_FOUND

    btfss   FAT_flags1, flags1_FAT_NO_MBR
    goto    m001
    
    DISPLAY_STR STR_CRLFPROMPT    
    DISPLAY_STR STR_NO_MBR

m001
    DISPLAY_STR STR_CRLFPROMPT
        
    
    ; Now, enter a loop awaiting commands over the RS232
    ; When a command is received, action it
mloop    
    DISPLAY_STR STR_CRLFPROMPT

    call    getCmdRS232   
    call    actionCommand

    goto    mloop
    
    
    
;*****************************************************************************        
;
;   Function :  AppHWInit
;               Perform the hardware initialisation required by the application
;
;   Input:      None.
;
;   Output:     None
;
;*****************************************************************************        
AppHWInit

    clrf    BSR
    
    ; Ensure all unused I/O pins are set to outputs
    
    ; PortA as I/O pins
    movlw   0x07    ; Disconnect comparitor from I/O pins
    movwf   CMCON 
    
    movlw   0x00    ; reference voltage not used
    movwf   CVRCON 
    
    movlw   0x0F    ; Disconnect A/D from I/O pins
    movwf   ADCON1 
    
    clrf    TRISA
    clrf    PORTA
    clrf    TRISB
    clrf    PORTB
    
    ; The main application will call an SPI init routine to set up the 
    ; SPI hardware requirements, and we will use the default bootloader RS232 
    ; interface for user I/O

    ; Set our default buffer block size, which is 512 bytes    
    movlw   0x01    
    movwf   blockSize       
        
    return
    
    
        
;*****************************************************************************        
;
;   Function :  getCmdRS232
;               Reads a command ( up to 31 characters long ) from the
;               serial port. Terminated by a CR or LF character
;
;   Input:      None.
;
;   Output:     cmdLine populated with the text, zero terminated
;
;*****************************************************************************        
getCmdRS232
    lfsr    FSR0, cmdLine
    clrf    index
    
    
    ; Zero the string first
    clrf    INDF0
    
    ; read characters, copy to string until CR, LF or length = 31
gcr001    
    call    RS232RxByte 
    
    ; If we have a CR or LF character, we end
    movf    hostByte, W
    sublw   0x0A
    btfsc   STATUS, Z
    return

    movf    hostByte, W
    sublw   0x0D
    btfsc   STATUS, Z
    return
    
    ; Save character, and add zero terminator
    movf    hostByte, W
    movwf   POSTINC0
    clrf    INDF0
    incf    index, F
    
    ; are we at the end of the string?
    movlw   D'31'
    subwf   index, W
    btfss   STATUS, Z
    goto    gcr001
        
    return
    
    

;*****************************************************************************        
;
;   Function :  actionCommand
;               Scans the cmdLine string for a command, and calls the 
;               appropriate code. 
;
;   Input:      Uses COMMAND_TABLE to identify commands and functions
;
;   Output:     None.
;
;*****************************************************************************        
actionCommand

    movlw   HIGH COMMAND_TABLE
    movwf   cmdPtrH
    movlw   LOW  COMMAND_TABLE
    movwf   cmdPtrL    
    

ac001
    clrf    TBLPTRU
    movf    cmdPtrH, W
    movwf   TBLPTRH
    movf    cmdPtrL, W
    movwf   TBLPTRL    

    tblrd*+
    movf    TABLAT, W
    movwf   codePtrL
    tblrd*+
    movf    TABLAT, W
    movwf   codePtrH
    
    ; If this pointer is to null, exit
    iorwf   codePtrL, W
    btfsc   STATUS, Z
    return
    
    ;
    ; codePtr points to a command string. Is it the same as ours?
    ;
    
    ; FSR0 points to the cmd string entered by user,
    ; tblrd set to point to the beginning of the rom string
    lfsr    FSR0, cmdLine
    
    movf    codePtrH, W
    movwf   TBLPTRH
    movf    codePtrL, W
    movwf   TBLPTRL
    
acNextChar    
    tblrd*+
    movf    TABLAT, W
    
    ; If it is null, we have reached the end of the rom string without any
    ; error, so execute the command.
    btfsc   STATUS, Z
    goto    acDoAction
    
    ; Ignore the bit that switches between upper and lower case
    xorwf   POSTINC0, W
    andlw   0xDF
    
    btfss   STATUS, Z
    goto    acNextCmd               ; Command chars differs - try next command
    
    goto    acNextChar
    
    
acNextCmd   
    ; move to next string pointer
    movlw   0x06
    addwf   cmdPtrL, F
    movlw   0x00
    addwfc  cmdPtrH, F
    
    goto    ac001
    
acDoAction
    ; Jump to the function
    
    movlw   0x04
    addwf   cmdPtrL, F
    movlw   0x00
    addwfc  cmdPtrH, W
    movwf   TBLPTRH
    movf    cmdPtrL, W
    movwf   TBLPTRL
    
    tblrd*+
    movf    TABLAT, W
    movwf   codePtrL
    tblrd*+
    movf    TABLAT, W
    movwf   PCLATH
    clrf    PCLATU
    movf    codePtrL, W
    movwf   PCL
    
    ; No more code - the last instruction magically jumps away
    ; to the select routine. When that returns, returns to the routine
    ; which called this one.
    
   
    
;*****************************************************************************        
;
;   Function :  ActionDspBuffer
;               Displays the size of the block setting, then dumps the buffer
;               - all 512 bytes, even if only 1 byte is set
;               This is only useful during debugging.
;
;   Input:      None.
;
;   Output:     None
;
;*****************************************************************************        
ActionDspBuffer
    DISPLAY_STR STR_CRLFPROMPT
    
    ; Display the block setting
    
    movf    blockSize, W
    btfss   STATUS, Z
    goto    adb001
    
    DISPLAY_STR STR_BSIZE1
    goto    adb002
    
adb001
    DISPLAY_STR STR_BSIZE512
    
adb002
    ; Now dump the 512 bytes, 32 bytes per line

    ; Setup indirect register to point to the buffer
    lfsr    FSR0, BLOCK_BUFFER_ADDRESS
    
    movlw   D'16'
    movwf   row
    
adb003    
    DISPLAY_STR STR_CRLFPROMPT

    movlw   D'32'
    movwf   column

adb004
    movf    POSTINC0, W
    call    DspHexByte    
    
    decfsz  column, F
    goto    adb004
    
    decfsz  row, F
    goto    adb003

    return    
    


;*****************************************************************************        
;
;   Function :  CmdDir
;               Displays the contents of the file system on the media
;
;   Input:      string pointed to by FSR0, with command text removed
;
;   Output:     Bytes written over rs232
;
;*****************************************************************************        
CmdDir
    
    ; Open up the directory listing. This just sets the variables
    ; to start from the beginning of the directory list
    call    FAT16StartDirList
    
    ; Display a table banner ( filename, file size )
    DISPLAY_STR STR_CRLFPROMPT  
    DISPLAY_STR STR_CRLFPROMPT
    DISPLAY_STR STR_DIR_HEAD
    DISPLAY_STR STR_CRLFPROMPT
    
    ; We use this to count the length of each filename, so we can
    ; display the next field aligned with those below
    clrf    index                       
    
cd001
    ; Get the next file in the directory. Check to see if we
    ; have reached the end of the list
    call    FAT16NextDirList
    btfsc   STATUS, C    
    return    
    
    ; We have a valid directory entry, 32 bytes at location
    ; FAT_Dir_Entry. Display the filename and filesize
    
    movlw   0x08
    movwf   tmpCount
    lfsr    FSR1, FAT_Dir_Entry 

cd002
    movf    INDF1, W
    sublw   ' '
    btfsc   STATUS, Z
    goto    cd003
    incf    index, F
    movf    POSTINC1, W         
    call    RS232TxByte
    decfsz  tmpCount, F
    goto    cd002    
    
cd003
    movlw   '.'
    call    RS232TxByte
    movlw   0x03
    movwf   tmpCount
    lfsr    FSR1, FAT_Dir_Entry+8
    
cd004
    movf    INDF1, W
    sublw   ' '
    btfsc   STATUS, Z
    goto    cd005
    incf    index, F
    movf    POSTINC1, W         
    call    RS232TxByte
    decfsz  tmpCount, F
    goto    cd004    
    
cd005    
    movf    index, W
    sublw   D'20'   
    movwf   index
    
cd006    
    movlw   ' '
    call    RS232TxByte
    decfsz  index, F
    goto    cd006
    
    ; Now print the file size
    
    movlw   '0'
    call    RS232TxByte
    movlw   'x'
    call    RS232TxByte
    movf    FAT_Dir_Entry+0x1F, W
    call    DspHexByte
    movf    FAT_Dir_Entry+0x1E, W
    call    DspHexByte
    movf    FAT_Dir_Entry+0x1D, W
    call    DspHexByte
    movf    FAT_Dir_Entry+0x1C, W
    call    DspHexByte
    
    
    DISPLAY_STR STR_CRLFPROMPT
    
    goto    cd001
    
        

;*****************************************************************************        
;
;   Function :  CmdWrite
;               Creates a file with 512 bytes of data in it
;
;   Input:      string in CMDLine
;
;   Output:     Media updated
;
;*****************************************************************************        
CmdWrite
    ; Get the filename that the user typed.
    call    getFileNameFAT
    
    ; Create the file in the filesystem
    call    FAT16OpenFileWrite
    
    ; Generate some dummy text data to fill the file with
    
    lfsr    FSR0, BLOCK_BUFFER_ADDRESS
    movlw   0x02
    movwf   indH
    movlw   0xFF
    movwf   indL
    movlw   'M'
cw001
    movwf   POSTINC0
    decfsz  indL, F
    goto    cw001
    decfsz  indH, F
    goto    cw001
    movlw   'E'
    movwf   POSTINC0
    
    ; Now write the data to the file. We write it multiple times
    ; to create a big file.
    call    FAT16WriteFile
    call    FAT16WriteFile
    call    FAT16WriteFile
    call    FAT16WriteFile
    call    FAT16WriteFile
    call    FAT16WriteFile
    call    FAT16WriteFile
    call    FAT16WriteFile
    call    FAT16WriteFile
    call    FAT16WriteFile
    call    FAT16WriteFile
    call    FAT16WriteFile
    call    FAT16WriteFile
    call    FAT16WriteFile
    call    FAT16WriteFile
    call    FAT16WriteFile
    call    FAT16WriteFile
    call    FAT16WriteFile
    
    ; Close the file. This is very important; closing the file
    ; updates the FAT table, marking the clusters that we have used.
    call    FAT16CloseFileWrite
    
    return
    
    
    
;*****************************************************************************        
;
;   Function :  CmdTypeFile
;               Displays the first 512 bytes of the specified file, if it exists
;
;   Input:      string in CMDLine
;
;   Output:     Bytes written over rs232
;
;*****************************************************************************        
CmdTypeFile
    call    getFileNameFAT
    
    call    FAT16OpenFileRead
    
    ; If an error occurred opening the file, just return
    btfsc   STATUS, C
    return
    
    ; Read from the file. This will dump 512 bytes into the 
    ; sector buffer at BLOCK_BUFFER_ADDRESS.
    call    FAT16ReadFile

    ; display the 512 bytes as text.
    lfsr    FSR0, BLOCK_BUFFER_ADDRESS
    
    movlw   0x02
    movwf   indH

ctf002
    clrf    indL

ctf003
    movf    POSTINC0, W
    call    RS232TxByte
    decfsz  indL, F
    goto    ctf003
    
    decfsz  indH, F
    goto    ctf002           
    
ctfExit
    return
    
    

;*****************************************************************************        
;
;   Function :  CmdHelpScreen
;               Displays the help text
;
;   Input:      None.
;
;   Output:     Bytes written over rs232
;
;*****************************************************************************        
CmdHelpScreen
    DISPLAY_STR STR_CRLFPROMPT
    DISPLAY_STR STR_HELP1
    DISPLAY_STR STR_CRLFPROMPT
    
    ; Display the list of commands from the COMMAND_TABLE
        
    movlw   HIGH (COMMAND_TABLE + 2)
    movwf   cmdPtrH
    movlw   LOW  (COMMAND_TABLE + 2)
    movwf   cmdPtrL    
    

cds001
    clrf    TBLPTRU
    movf    cmdPtrH, W
    movwf   TBLPTRH
    movf    cmdPtrL, W
    movwf   TBLPTRL    

    tblrd*+
    movf    TABLAT, W
    movwf   codePtrL
    tblrd*+
    movf    TABLAT, W
    movwf   codePtrH
    
    ; If this pointer is to null, exit
    iorwf   codePtrL, W
    btfsc   STATUS, Z
    return
        
    ; Display the string + CRLF
    movf    codePtrH, W
    movwf   TBLPTRH
    movf    codePtrL, W
    movwf   TBLPTRL
    
    call    DspRomStr 
    
    DISPLAY_STR STR_CRLFPROMPT
    
    ; move to next string pointer
    movlw   0x06
    addwf   cmdPtrL, F
    movlw   0x00
    addwfc  cmdPtrH, F
    
    goto    cds001
    
    

;*****************************************************************************        
;
;   Function :  getFileNameFAT
;               gets a filename from the cmdLine and stores it in FAT_Filename,
;               an 11 byte string, formatted exactly as per the FAT directory 
;               format.
;
;   Input:      CmdLine
;
;   Output:     FAT_Filename updated
;
;               NOTE: error checking should be done when parsing the line,
;               for example to make sure that no invalid characters have been
;               entered.
;
;*****************************************************************************        
getFileNameFAT
    ; First, init FAT_Filename to all spaces
    movlw   D'11'
    movwf   tmpCount
    lfsr    FSR1, FAT_Filename
    movlw   ' '
gfn001    
    movwf   POSTINC1
    decfsz  tmpCount, F
    goto    gfn001
    
    lfsr    FSR0, cmdLine
    lfsr    FSR1, FAT_Filename
    
    ; skip past the command, to the first space character or NULL
gfn002    
    movf    POSTINC0, W
    
    ; The check for NULL
    btfsc   STATUS, Z
    goto    gfnExit 
    
    movwf   tmpByte
    sublw   ' '
    btfss   STATUS, Z
    goto    gfn002
    
    clrf    tmpCount        ; counts # of characters copied 
   
gfn003
    movf    POSTINC0, W
    movwf   tmpByte

    
    ; If '.', move to next set.
    ; If NULL, exit

    sublw   '.'
    btfsc   STATUS, Z
    goto    gfnExt
    
    movf    tmpByte, W
    btfsc   STATUS, Z
    goto    gfnExit 
    
    ; Store the byte
    movwf   POSTINC1
    
    ; If 8 chars, move to next set
    incf    tmpCount, F
    movlw   8
    subwf   tmpCount, W
    btfss   STATUS, Z
    goto    gfn003
    
    ; Get the next char. if NULL, end
    movf    POSTINC0, W
    btfsc   STATUS, Z
    goto    gfnExit 
    
gfnExt
    movf    POSTINC0, W
    btfsc   STATUS, Z
    goto    gfnExit
     
    movwf   FAT_Filename + 8
    movf    POSTINC0, W
    btfsc   STATUS, Z
    goto    gfnExit
     
    movwf   FAT_Filename + 9
    movf    POSTINC0, W
    btfsc   STATUS, Z
    goto    gfnExit
    
    movwf   FAT_Filename + D'10'  
    
gfnExit
    ; Change lower case characters to upper case
          
    lfsr    FSR1, FAT_Filename
    movlw   D'11'
    movwf   tmpCount
    
gfn004
    movf    INDF1, W
    andlw   0x40
    btfsc   STATUS, Z
    goto    gfn005    
    
    movf    INDF1, W
    andlw   0xDF
    movwf   INDF1

gfn005    
    movf    POSTINC1, W     ; Dummy; just used to increment value
    decfsz  tmpCount, F
    goto    gfn004
    return    
    
    
 
;*****************************************************************************        
;
;   Function :  DspRomStr
;               Writes a null terminated string, stored in code space, to the
;               rs232 port.
;
;               The strings should be defines using the DA derective, storing
;               2 bytes per word location.
;
;   Input:      TBLPTR registers point to begining of string
;
;   Output:     bytes written
;
;*****************************************************************************        
DspRomStr
    tblrd*+
    movf    TABLAT, W
    btfsc   STATUS, Z
    return
    call    RS232TxByte
    goto    DspRomStr



;*****************************************************************************        
;
;   Function :  DspHexByte
;               Write a 2 digit hex byte in uppercase to the user interface
;
;   Input:      byte to display in W
;
;   Output:     2 characters written to the UI
;
;*****************************************************************************        
DspHexByte    
    movwf   tmpHexByte
    swapf   tmpHexByte,W
    sublw   0x09
    swapf   tmpHexByte,W
    andlw   0x0F
    btfss   STATUS,DC
    addlw   'A' - .10 - '0'
    addlw   '0'
    call    RS232TxByte
    movf    tmpHexByte, W
    sublw   0x09
    movf    tmpHexByte, W
    andlw   0x0F
    btfss   STATUS,DC
    addlw   'A' - .10 - '0'
    addlw   '0'
    call    RS232TxByte
    
    return
    
    

;*****************************************************************************        
;
; STRING CONSTANTS
;
;*****************************************************************************        

STR_MAIN        DA  "EPE MMC/FAT Interface v1.0", 0x00
STR_CRLF        DA   0x0a0d, 0x00
STR_CRLFPROMPT  DA   0x0a0d, "> ", 0x00
STR_HELP1       DA  "The following commands are supported:", 0x00
STR_BSIZE1      DA  "Block Size is 1 byte", 0x00
STR_BSIZE512    DA  "Block Size is 512 bytes", 0x00
STR_CID         DA  "Card CID is ",0x00
STR_DIR_HEAD    DA  "Filename             Size (in hex)",0x00

; Strings displayed for debugging purposes.
; You may remove any printout code in this file if you wish.
STR_FAT_NO          DA "No ", 0x00
STR_FAT_FOUND       DA "FAT file system found.", 0x00
STR_NO_MBR          DA "No MBR found - partition starts in sector 0.", 0x00

; These strings define what the user has to type to get a command to be executed
CMD_HELP        DA  "HELP", 0x00
CMD_TYPE        DA  "TYPE", 0x00
CMD_DIR         DA  "DIR",  0x00
CMD_WRITE       DA  "WRITE",  0x00
CMD_BUFF        DA  "BUFF",  0x00

; These strings explain what each command does
STR_HELP        DA  " HELP  Display this screen",0x00
STR_TYPE        DA  " TYPE filename.ext  Display content of file",0x00
STR_WRITE       DA  " WRITE filename.ext  Writes 512 bytes to a file",0x00
STR_DIR         DA  " DIR  List files on media",0x00
STR_BUFF        DA  " BUFF display the content of the sector buffer", 0x00



;*****************************************************************************        
;
; COMMAND TABLE
;
; This list of data provides a flexible way to create a user interface.
; The table consists of a series of 3 WORD values, which represent a command
; that can be entered by the user. The first word points to the string that 
; is the command the user typed; The second, a pointer to the string that
; describes the command on a help screen, and the third is a pointer to the
; actual command executed when that command is typed by the user.
; The function handle_command takes a null terminated string (entered by the
; user via the RS232 interface), parses it and calls the appropriate command 
; as listed here.
;
;*****************************************************************************        

COMMAND_TABLE   DW  CMD_HELP, STR_HELP, CmdHelpScreen
                DW  CMD_TYPE, STR_TYPE, CmdTypeFile
                DW  CMD_DIR,  STR_DIR,  CmdDir
                DW  CMD_WRITE,  STR_WRITE,  CmdWrite
                DW  CMD_BUFF, STR_BUFF, ActionDspBuffer
                DW  0x0000,   0x0000,   0x0000 ; End of commands marker