;*****************************************************************************        
;
;   Module:     interrupts.inc
;               
;   Author:     Mike Hibbett 
;                                                                  
;   Version:    0.1 10/06/05                                                  
;
;               The re-mapped interrupt routines. This file will start at
;               0x608 and MUST not be moved, unless you change sw2.asm 
;
;
;*****************************************************************************        


;*****************************************************************************        
;
;   Function :  IntVecHigh
;               The re-mapped fast interrupt vector goes at 0x608
;
;   Input:      None.
;
;   Output:     Using the FAST directive to update registers automatically
;
;*****************************************************************************        
IntVecHigh
    btfss   INTCON, TMR0IF
    bra     IVH001

    bcf     INTCON, TMR0IF
    movf    tmr0ValH, W
    movwf   TMR0H
    movf    tmr0ValL, W
    movwf   TMR0L
    
    btg     LATC, 2     ; Toggle sound pin

IVH001
    btfss   PIR1, RCIF
    bra     TmrInt
    
    ; Check for receive error
    movf    RCSTA, W
    andlw   0x06
    bz      M001
    
    ; A receive error occurred:
    
    ; Clear the error
    bcf     RCSTA, CREN
    bcf     RCSTA, OERR
    bcf     RCSTA, FERR
    bsf     RCSTA, CREN
    
    ; Increment error count, stop at 255 
    incfsz  rxErrCount, F
    decf    rxErrCount, F

M001
    movf    PCLATH, W
    movwf   intPCLATH
    
    movf    RCREG, W
    call    RxStateMachine  
    
    movf    intPCLATH, W
    movwf   PCLATH
        
TmrInt    
    btfss   PIR2, CCP2IF
    retfie  FAST
    
    bcf     PIR2, CCP2IF
    
    ; decrement any active general purpose timers
    tstfsz  intTmr1
    decf    intTmr1, F
;    tstfsz  appTmpDspTmr
;    decf    appTmpDspTmr, F
    tstfsz  appDspTmr
    decf    appDspTmr, F
    
    ; decrement the keypress debounce timers
    tstfsz  keyNextTmr
    decf    keyNextTmr, F
    tstfsz  keySelectTmr
    decf    keySelectTmr, F
    tstfsz  keyClearTmr
    decf    keyClearTmr, F
    tstfsz  keyRecordTmr
    decf    keyRecordTmr, F
        
    ; Test the keys. A pin is low when pressed
    ; Ignore the keys if already detected and still down.    
    btfss   PORTA, KEY_NEXT
    call    keyNextPressed
    btfss   PORTA, KEY_SELECT
    call    keySelectPressed
    btfss   PORTA, KEY_CLEAR
    call    keyClearPressed
    btfss   PORTA, KEY_RECORD
    call    keyRecordPressed
    
    ; If any keys are released, make sure we clear the pressed flag
    btfsc   PORTA, KEY_NEXT
    bcf     keyStates, KEY_NEXT
    btfsc   PORTA, KEY_SELECT
    bcf     keyStates, KEY_SELECT
    btfsc   PORTA, KEY_CLEAR
    bcf     keyStates, KEY_CLEAR
    btfsc   PORTA, KEY_RECORD
    bcf     keyStates, KEY_RECORD
        
    ; Test the melody generator state machine
    goto    MelodyMaker     ; NOTE: No code can follow this!
                            ; MelodyMaker will do the retfie FAST

    

;*****************************************************************************        
;
;   Function :  UserIntVecLow
;               Low level interrupt handler
;
;   Input:      None.
;
;   Output:     MUST not use FAST return directive, since the high level 
;               interrupt uses this.
;
;*****************************************************************************        
UserIntVecLow
    reset
    retfie




;*****************************************************************************        
;
;   Function :  Keypress routines
;               Called from interrupt routine to handle key debouncing
;
;   Input:      None.
;
;   Output:     None.
;
;*****************************************************************************        

keyNextPressed
    ; If the key is already marked as active (1), ignore it
    btfsc   keyStates, KEY_NEXT
    return
    
    ; If the 'ignore key timer is running, well, ignore it!
    tstfsz  keyNextTmr
    return    
    
    ; OK, it's a new key down event.
    
    ; Indicate it to the application
    bsf     flags2, KEY_NEXT_FLAG2
    
    ; Indicate it to the interrupt routine
    bsf     keyStates, KEY_NEXT
    
    ; start the 'ignore key' timer
    movlw   KEY_NEXT_TIMER_VAL
    movwf   keyNextTmr
    
    return
    


keySelectPressed
    ; If the key is already marked as active (1), ignore it
    btfsc   keyStates, KEY_SELECT
    return
    
    ; If the 'ignore key timer is running, well, ignore it!
    tstfsz  keySelectTmr
    return    
    
    ; OK, it's a new key down event.
    
    ; Indicate it to the application
    bsf     flags2, KEY_SELECT_FLAG2
    
    ; Indicate it to the interrupt routine
    bsf     keyStates, KEY_SELECT
    
    ; start the 'ignore key' timer
    movlw   KEY_SELECT_TIMER_VAL
    movwf   keySelectTmr
    
    return



keyClearPressed
    ; If the key is already marked as active (1), ignore it
    btfsc   keyStates, KEY_CLEAR
    return
    
    ; If the 'ignore key timer is running, well, ignore it!
    tstfsz  keyClearTmr
    return    
    
    ; OK, it's a new key down event.
    
    ; Indicate it to the application
    bsf     flags2, KEY_CLEAR_FLAG2
    
    ; Indicate it to the interrupt routine
    bsf     keyStates, KEY_CLEAR
    
    ; start the 'ignore key' timer
    movlw   KEY_CLEAR_TIMER_VAL
    movwf   keyClearTmr
    
    return



keyRecordPressed
    ; If the key is already marked as active (1), ignore it
    btfsc   keyStates, KEY_RECORD
    return
    
    ; If the 'ignore key timer is running, well, ignore it!
    tstfsz  keyRecordTmr
    return    
    
    ; OK, it's a new key down event.
    
    ; Indicate it to the application
    bsf     flags2, KEY_RECORD_FLAG2
    
    ; Indicate it to the interrupt routine
    bsf     keyStates, KEY_RECORD
    
    ; start the 'ignore key' timer
    movlw   KEY_RECORD_TIMER_VAL
    movwf   keyRecordTmr
    
    return
   