;*****************************************************************************        
;
;   Module:     displays.inc
;               
;   Author:     Mike Hibbett 
;                                                                  
;   Version:    0.1 28/06/05                                            
;
;               Handler for the user interface display
;
;               Each display has it's own routine, which is a mini state-machine
;               HandleDisplay is called by the main loop to manage the display
;               of the appropriate routine
;
;*****************************************************************************        


;*****************************************************************************        
;
;   Function :  HandleDisplay
;               jumps to the currently active display
;
;   Input:      None
;
;   Output:     None - Return provided by display routine
;
;*****************************************************************************        

; Display Routine Handler constants
NUM_DISPLAYS            EQU 0x0C    ; Number of routines (goto lines) in HandleDisplay

 ORG 0x1880

HandleDisplay
    movlw   high HandleDisplay
    movwf   PCLATH
    ; times by 4 to get offset
    movf    defaultDisplay, W
    addwf   defaultDisplay, W
    addwf   defaultDisplay, W
    addwf   defaultDisplay, W
    
    addwf   PCL, F

    goto    AppDspBarGraph  
    
    ifdef ( MILES ) 
     goto    AppDsp30Limit      
     goto    AppDsp40Limit      
     goto    AppDsp50Limit      
     goto    AppDsp60Limit      
     goto    AppDsp70Limit    
    else  
     goto    AppDsp30KLimit      
     goto    AppDsp50KLimit      
     goto    AppDsp80KLimit      
     goto    AppDsp100KLimit      
     goto    AppDsp120KLimit      
    endif
    
    goto    AppDspSpeed
    goto    AppDspAltHeading
    goto    AppDspPosition
    goto    AppDspSWVersionInfo     
    goto    AppDspConnectPC
    goto    AppDspConnectedPC
    
HandleDisplay_end
    
    IF ( (HandleDisplay & 0x0FF) >= (HandleDisplay_end & 0x0FF) )
        MESSG   "Table HandleDisplay overflow"
    ENDIF



;*****************************************************************************        
;
;   Function :  AppDspPosition
;               Display the lat/long position, as received from the module
;               Format is
;                dd mm.mmmm N
;               ddd mm.mmmm E
;               Only update it once per second; This display may be called 
;               more frequently, so use the application display timer to 
;               manage the update rate
;
;   Input:      Nothing specific
;
;   Output:     N/A
;
;*****************************************************************************        
AppDspPosition  
    tstfsz  appDspTmr
    return

    ; Restart the display update timer    
    movlw   D'20' 
    movwf   appDspTmr
    
    call    DspHome
    call    DspClear

    movf    latText, W, BANKED
    call    DspPutChar
    movf    latText+1, W, BANKED
    call    DspPutChar
    movf    latText+2, W, BANKED
    call    DspPutChar
    movf    latText+3, W, BANKED
    call    DspPutChar
    movf    latText+4, W, BANKED
    call    DspPutChar
    movf    latText+5, W, BANKED
    call    DspPutChar
    movf    latText+6, W, BANKED
    call    DspPutChar
    movf    latText+7, W, BANKED
    call    DspPutChar
    movf    latText+8, W, BANKED
    call    DspPutChar
    movf    latText+9, W, BANKED
    call    DspPutChar
    movf    latText+0x0A, W, BANKED
    call    DspPutChar
    
    call    DspLine2    

    movf    longText, W, BANKED
    call    DspPutChar
    movf    longText+1, W, BANKED
    call    DspPutChar
    movf    longText+2, W, BANKED
    call    DspPutChar
    movf    longText+3, W, BANKED
    call    DspPutChar
    movf    longText+4, W, BANKED
    call    DspPutChar
    movf    longText+5, W, BANKED
    call    DspPutChar
    movf    longText+6, W, BANKED
    call    DspPutChar
    movf    longText+7, W, BANKED
    call    DspPutChar
    movf    longText+8, W, BANKED
    call    DspPutChar
    movf    longText+9, W, BANKED
    call    DspPutChar
    movf    longText+0x0A, W, BANKED
    call    DspPutChar

    return
    


;*****************************************************************************        
;
;   Function :  AppDspSWVersionInfo
;               Display the device software version
;
;   Input:      Nothing specific
;
;   Output:     N/A
;
;*****************************************************************************        
AppDspSWVersionInfo  
    tstfsz  appDspTmr
    return

    ; Restart the display update timer    
    movlw   D'20'
    movwf   appDspTmr

    call    DspHome
    call    DspClear

    DISPLAY STRSWVL1
    call    DspLine2    
    DISPLAY STRSWVL2
    
    return
    
    

;*****************************************************************************        
;
;   Function :  AppDspResetMemory
;               Enable the user to clear all the memory
;
;   Input:      Nothing specific
;
;   Output:     N/A
;
;*****************************************************************************        
;AppDspResetMemory  
;    tstfsz  appDspTmr
;    return
;
;    ; Restart the display update timer    
;    movlw   D'20'
;    movwf   appDspTmr
;
;    call    DspHome
;    call    DspClear
;
;    DISPLAY STRResetMemoryL1
;    call    DspLine2    
;    DISPLAY STRResetMemoryL2
;    
;    return
;
;ADRM001
;    bcf     flags2, KEY_SELECT_FLAG2
;
;    ; Clear the memory
;    call    EEPClear
;    
;    ; 'reload' the defaults
;    call    GetSettings
;
;    ; Restart the display update timer    
;    movlw   D'20'
;    movwf   appDspTmr
;
;    ; Make display return to the default
;    clrf    eepAddH
;    movlw   0x04
;    movwf   eepAddL
;    call    EEPRead
;    movf    eepData, W
;    movwf   defaultDisplay
;    
;    call    DspHome
;    call    DspClear
;
;    DISPLAY STRMemoryResetL1
;    call    DspLine2    
;    DISPLAY STRMemoryResetL2
;
;    return
;    
;
;
;;*****************************************************************************        
;;
;;   Function :  AppDspRecordsInfo
;;               Display the number of entries in the database
;;
;;   Input:      Nothing specific
;;
;;   Output:     N/A
;;
;*****************************************************************************        
;AppDspRecordsInfo
;    btfsc   flags2, KEY_SELECT_FLAG2
;    call    SelectKeyPressed
;       
;    tstfsz  appDspTmr
;    return
;
;    ; Restart the display update timer    
;    movlw   D'20'
;    movwf   appDspTmr
;
;    call    DspHome
;    call    DspClear
;
;    movf    numEntriesH, W
;    call    DspHexByte
;
;    movf    numEntriesL, W
;    call    DspHexByte
;
;    movlw   ' '
;    call    DspPutChar
;
;    DISPLAY STREntries
;    
;    return



   



;*****************************************************************************        
;
;   Function :  AppDspConnectPC
;               Prompts user to select PC link, then provides the PC interface
;               feature. This allows for eeprom loading/downloading
;
;   Input:      Nothing specific
;
;   Output:     A reboot occurrs
;
;*****************************************************************************        
AppDspConnectPC
    tstfsz  appDspTmr
    return

    ; Just refresh the display and return.
    ; Restart the display update timer    
    movlw   D'255'
    movwf   appDspTmr

    call    DspHome
    call    DspClear

    DISPLAY STRConnectPCL1
    call    DspLine2    
    DISPLAY STRConnectPCL2

    return
    
    
    
    

;*****************************************************************************        
;
;   Function :  AppDspConnectedPC
;               Enters PC conenction mode. Exited by reset only
;
;   Input:      Nothing specific
;
;   Output:     A reboot occurrs
;
;*****************************************************************************        
AppDspConnectedPC

    ; First, must make sure that the unit does not remember this 
    ; display index!
    clrf    eepData
    clrf    eepAddH
    movlw   0x04
    movwf   eepAddL
    call    EEPWrite           


    ; Bit-bashed RS232 comms dont work well with interrupts!    
    bcf     INTCON, GIE
    bcf     LATC, 2     ; Ensure not drawing current through speaker
    call    DspHome
    call    DspClear

    DISPLAY STRConnectedPCL1
        
    ; Sync with the PC by exchanging 'X' bytes, followed by a G for Go!
ADCP003
    movlw   0x01
    call    UIWait10ms
    movlw   'X'
    call    RS232TxByte
    call    RS232RxByte
    btfsc   flags2, KEY_NEXT_FLAG2
    bra     ADCPExit
    movlw   'X'
    subwf   hostByte, W
    btfss   STATUS, Z
    bra     ADCP003
       
    call    DspHome
    call    DspClear
    DISPLAY STRCommsPCL1       
       
    ; Signal to the host that it may start - 'G' for Go!
    movlw   D'20'
    call    UIWait50us
    movlw   'G'
    call    RS232TxByte

    ; Then wait for a R or W or E(xit) or the Next button.
ADCP004    
    call    RS232RxByte
    btfsc   flags2, KEY_NEXT_FLAG2
    bra     ADCPExit

    movlw   'R'
    subwf   hostByte, W
    btfsc   STATUS, Z
    goto    ADCPRead
    
    movlw   'W'
    subwf   hostByte, W
    btfss   STATUS, Z
    bra     ADCPExit        ; Any other response indicates we have finished.
    
    ; Write
    ;======
    movf    hostByte, W
    movwf   progRxCheck
    call    RS232TxByte

    ; Echo Y OK or C checksum error
    
    call    RS232RxByte
    movf    hostByte, W
    movwf   eepAddH
    addwf   progRxCheck, F
    call    RS232TxByte
    call    RS232RxByte
    movf    hostByte, W
    movwf   eepAddL
    addwf   progRxCheck, F
    call    RS232TxByte

    ; Receive and store the data
    
    movlw   D'64'
    movwf   TEMP
    
    ; Stick the 64 bytes high up out of the way. At 0x2B0
    movlw   0xB0
    movwf   FSR0L
    movlw   0x02
    movwf   FSR0H
    
ADCRWLoop1
    call    RS232RxByte
    movf    hostByte, W
    movwf   POSTINC0
    addwf   progRxCheck, F
    call    RS232TxByte
    
    decfsz  TEMP, F
    bra     ADCRWLoop1
    
    call    RS232RxByte
    movf    hostByte, W
    subwf   progRxCheck, W
    btfss   STATUS, Z
    bra     ADCRWErr          ; CheckSum Error 
    
    ; Checksum ok - program the bytes
    movlw   0xB0
    movwf   FSR0L
    movlw   0x02
    movwf   FSR0H
    
    ; write the 64 bytes pointed to by FSR0
    call    EEPWrite64
    
    ; Send back response Y, programmed ok
    movlw   'Y'
    call    RS232TxByte
    goto    ADCP004

ADCRWErr
    ; Send back response
    movlw   0x01
    call    UIWait10ms
    movlw   'C'
    call    RS232TxByte
    goto    ADCP004
        
ADCPRead    
    ; Read
    ; ====
    movf    hostByte, W
    movwf   progRxCheck  
    call    RS232TxByte

    ; rx 2 byte address; Return 64 bytes & checksum
    call    RS232RxByte
    movf    hostByte, W
    movwf   eepAddH
    addwf   progRxCheck, F
    call    RS232TxByte
    call    RS232RxByte
    movf    hostByte, W
    movwf   eepAddL
    addwf   progRxCheck, F
    call    RS232TxByte

    movlw   D'20'
    call    UIWait50us
    
    ; Read 64 bytes, and output them, then the checksum
    
    movlw   D'64'
    movwf   TEMP        ; Use temp as the loop counter

ADCRRLoop    
    call    EEPRead

    movf    eepData, W
    addwf   progRxCheck, F
    call    RS232TxByte
    call    RS232RxByte     ; Wait for PC to respond
 
    infsnz  eepAddL, F
    incf    eepAddH, F
    
    decfsz  TEMP, F
    bra     ADCRRLoop
    
    movf    progRxCheck, W
    call    RS232TxByte
            
    goto    ADCP004

ADCPExit
    bsf     INTCON, GIE

    bcf     flags2, KEY_NEXT_FLAG2
    clrf    appDspTmr
    reset                   ; Force a reboot, loading new data from eeprom
    return 



;*****************************************************************************        
;
;   Function :  AppDspBarGraph
;               Displays the current closest camera as a bargraph
;
;
;   Input:      Nothing specific
;
;   Output:     N/A
;
;*****************************************************************************        
AppDspBarGraph
    tstfsz  appDspTmr
    return

    ; Restart the display update timer    
    movlw   D'10'
    movwf   appDspTmr

    call    DspHome

    DISPLAY STRDistance

    call    DspLine2    

    btfss   flags2, INLOCKFLAG2
    goto    ADBG001
    
    ; Display the distance as a bar graph
    call    DspBarGraph
        
    return

ADBG001    
    ; Just tell the user there is no signal
    DISPLAY STRNoLock

    return        



;*****************************************************************************        
;
;   Function :  AppDsp30Limit
;               Displays the current closest camera as a bargraph and
;               also signals when the speed exceeds 30mph
;
;
;   Input:      Nothing specific
;
;   Output:     N/A
;
;*****************************************************************************        
AppDsp30Limit
    tstfsz  appDspTmr
    return

    ; Restart the display update timer    
    movlw   D'10'
    movwf   appDspTmr

    ; If we are moving faster than 30MPH, start to play the warning alert.
    ; Otherwise, turn it off.
    
    
    ; The test: current speed - 30MPH
    ; if result is postive, ie carry is set we are not speed
    ;
    ; If we are speeding and we were not before, then set flags1 bit 
    ; and start alert.
    ; If we are not speeding and we were before, clear ALL alert bits
    ; in flag1 and stop alert ( this ensures any other alert, eg camera, can
    ; restart
    
    movlw   LOW SPEED_30MPH
    subwf   SpeedL, W
    movlw   HIGH SPEED_30MPH
    subwfb  SpeedH, W
    btfss   STATUS, C 
    goto    AD30L000
    
    ; exceeding the speed.
    ; Are we already exceeding the speed? Goto AD30L002 if so
    ; and start alert if not
    btfsc   flags1, OVERSPEED_FLAG1
    goto    AD30L002
    
    bsf     flags1, OVERSPEED_FLAG1
    movlw   ALERT_OVER_SPEED
    movwf   tuneCurrent
    goto    AD30L002
    
AD30L000    
    ; Not exceeding the speed
    ; were we exceeding the speed? Goto AD30L002 if not
    ; and start alert if not
    btfss   flags1, OVERSPEED_FLAG1
    goto    AD30L002
    
    bcf     flags1, OVERSPEED_FLAG1
    movlw   ALERT_SILENCE
    movwf   tuneCurrent
    bcf     flags1, MASK_ALERT_FLAG1
   
AD30L002
    call    DspHome

    DISPLAY STR30MPH

    call    DspLine2   
    
    ; Do we have speed data yet? Warn user if not
    movf    SpeedH, W
    iorwf   SpeedL, W 
    btfss   STATUS, Z
    bra     AD30L003
    
    DISPLAY STRNoSpeed
    return

AD30L003
    btfss   flags2, INLOCKFLAG2
    goto    AD30L001
    
    ; Display the distance as a bar graph
    call    DspBarGraph
        
    return

AD30L001    
    ; Just tell the user there is no signal
    DISPLAY STRNoLock

    return        



;*****************************************************************************        
;
;   Function :  AppDsp40Limit
;               Displays the current closest camera as a bargraph and
;               also signals when the speed exceeds 40mph
;
;
;   Input:      Nothing specific
;
;   Output:     N/A
;
;*****************************************************************************        
AppDsp40Limit
    tstfsz  appDspTmr
    return

    ; Restart the display update timer    
    movlw   D'10'
    movwf   appDspTmr

    ; If we are moving faster than 40MPH, start to play the warning alert.
    ; Otherwise, turn it off.
    
    
    ; The test: current speed - 40MPH
    ; if result is postive, ie carry is set we are not speed
    ;
    ; If we are speeding and we were not before, then set flags1 bit 
    ; and start alert.
    ; If we are not speeding and we were before, clear ALL alert bits
    ; in flag1 and stop alert ( this ensures any other alert, eg camera, can
    ; restart
    
    movlw   LOW SPEED_40MPH
    subwf   SpeedL, W
    movlw   HIGH SPEED_40MPH
    subwfb  SpeedH, W
    btfss   STATUS, C 
    goto    AD40L000
    
    ; exceeding the speed.
    ; Are we already exceeding the speed? Goto AD40L002 if so
    ; and start alert if not
    btfsc   flags1, OVERSPEED_FLAG1
    goto    AD40L002
    
    bsf     flags1, OVERSPEED_FLAG1
    movlw   ALERT_OVER_SPEED
    movwf   tuneCurrent
    goto    AD40L002
    
AD40L000    
    ; Not exceeding the speed
    ; were we exceeding the speed? Goto AD40L002 if not
    ; and start alert if not
    btfss   flags1, OVERSPEED_FLAG1
    goto    AD40L002
    
    bcf     flags1, OVERSPEED_FLAG1
    movlw   ALERT_SILENCE
    movwf   tuneCurrent
    bcf     flags1, MASK_ALERT_FLAG1
   
AD40L002
    call    DspHome

    DISPLAY STR40MPH

    call    DspLine2   
    
    ; Do we have speed data yet? Warn user if not
    movf    SpeedH, W
    iorwf   SpeedL, W 
    btfss   STATUS, Z
    bra     AD40L003
    
    DISPLAY STRNoSpeed
    return

AD40L003
    btfss   flags2, INLOCKFLAG2
    goto    AD40L001
    
    ; Display the distance as a bar graph
    call    DspBarGraph
        
    return

AD40L001    
    ; Just tell the user there is no signal
    DISPLAY STRNoLock

    return        



;*****************************************************************************        
;
;   Function :  AppDsp50Limit
;               Displays the current closest camera as a bargraph and
;               also signals when the speed exceeds 50mph
;
;
;   Input:      Nothing specific
;
;   Output:     N/A
;
;*****************************************************************************        
AppDsp50Limit
    tstfsz  appDspTmr
    return

    ; Restart the display update timer    
    movlw   D'10'
    movwf   appDspTmr

    ; If we are moving faster than 50MPH, start to play the warning alert.
    ; Otherwise, turn it off.
    
    
    ; The test: current speed - 50MPH
    ; if result is postive, ie carry is set we are not speed
    ;
    ; If we are speeding and we were not before, then set flags1 bit 
    ; and start alert.
    ; If we are not speeding and we were before, clear ALL alert bits
    ; in flag1 and stop alert ( this ensures any other alert, eg camera, can
    ; restart
    
    movlw   LOW SPEED_50MPH
    subwf   SpeedL, W
    movlw   HIGH SPEED_50MPH
    subwfb  SpeedH, W
    btfss   STATUS, C 
    goto    AD50L000
    
    ; exceeding the speed.
    ; Are we already exceeding the speed? Goto AD50L002 if so
    ; and start alert if not
    btfsc   flags1, OVERSPEED_FLAG1
    goto    AD50L002
    
    bsf     flags1, OVERSPEED_FLAG1
    movlw   ALERT_OVER_SPEED
    movwf   tuneCurrent
    goto    AD50L002
    
AD50L000    
    ; Not exceeding the speed
    ; were we exceeding the speed? Goto AD50L002 if not
    ; and start alert if not
    btfss   flags1, OVERSPEED_FLAG1
    goto    AD50L002
    
    bcf     flags1, OVERSPEED_FLAG1
    movlw   ALERT_SILENCE
    movwf   tuneCurrent
    bcf     flags1, MASK_ALERT_FLAG1
   
AD50L002
    call    DspHome

    DISPLAY STR50MPH

    call    DspLine2   
    
    ; Do we have speed data yet? Warn user if not
    movf    SpeedH, W
    iorwf   SpeedL, W 
    btfss   STATUS, Z
    bra     AD50L003
    
    DISPLAY STRNoSpeed
    return

AD50L003
    btfss   flags2, INLOCKFLAG2
    goto    AD50L001
    
    ; Display the distance as a bar graph
    call    DspBarGraph
        
    return

AD50L001    
    ; Just tell the user there is no signal
    DISPLAY STRNoLock

    return        



;*****************************************************************************        
;
;   Function :  AppDsp60Limit
;               Displays the current closest camera as a bargraph and
;               also signals when the speed exceeds 60mph
;
;
;   Input:      Nothing specific
;
;   Output:     N/A
;
;*****************************************************************************        
AppDsp60Limit
    tstfsz  appDspTmr
    return

    ; Restart the display update timer    
    movlw   D'10'
    movwf   appDspTmr

    ; If we are moving faster than 60MPH, start to play the warning alert.
    ; Otherwise, turn it off.
    
    
    ; The test: current speed - 60MPH
    ; if result is postive, ie carry is set we are not speed
    ;
    ; If we are speeding and we were not before, then set flags1 bit 
    ; and start alert.
    ; If we are not speeding and we were before, clear ALL alert bits
    ; in flag1 and stop alert ( this ensures any other alert, eg camera, can
    ; restart
    
    movlw   LOW SPEED_60MPH
    subwf   SpeedL, W
    movlw   HIGH SPEED_60MPH
    subwfb  SpeedH, W
    btfss   STATUS, C 
    goto    AD60L000
    
    ; exceeding the speed.
    ; Are we already exceeding the speed? Goto AD60L002 if so
    ; and start alert if not
    btfsc   flags1, OVERSPEED_FLAG1
    goto    AD60L002
    
    bsf     flags1, OVERSPEED_FLAG1
    movlw   ALERT_OVER_SPEED
    movwf   tuneCurrent
    goto    AD60L002
    
AD60L000    
    ; Not exceeding the speed
    ; were we exceeding the speed? Goto AD60L002 if not
    ; and start alert if not
    btfss   flags1, OVERSPEED_FLAG1
    goto    AD60L002
    
    bcf     flags1, OVERSPEED_FLAG1
    movlw   ALERT_SILENCE
    movwf   tuneCurrent
    bcf     flags1, MASK_ALERT_FLAG1
   
AD60L002
    call    DspHome

    DISPLAY STR60MPH

    call    DspLine2   
    
    ; Do we have speed data yet? Warn user if not
    movf    SpeedH, W
    iorwf   SpeedL, W 
    btfss   STATUS, Z
    bra     AD60L003
    
    DISPLAY STRNoSpeed
    return

AD60L003
    btfss   flags2, INLOCKFLAG2
    goto    AD60L001
    
    ; Display the distance as a bar graph
    call    DspBarGraph
        
    return

AD60L001    
    ; Just tell the user there is no signal
    DISPLAY STRNoLock

    return        



;*****************************************************************************        
;
;   Function :  AppDsp70Limit
;               Displays the current closest camera as a bargraph and
;               also signals when the speed exceeds 70mph
;
;
;   Input:      Nothing specific
;
;   Output:     N/A
;
;*****************************************************************************        
AppDsp70Limit
    tstfsz  appDspTmr
    return

    ; Restart the display update timer    
    movlw   D'10'
    movwf   appDspTmr

    ; If we are moving faster than 70MPH, start to play the warning alert.
    ; Otherwise, turn it off.
    
    
    ; The test: current speed - 70MPH
    ; if result is postive, ie carry is set we are not speed
    ;
    ; If we are speeding and we were not before, then set flags1 bit 
    ; and start alert.
    ; If we are not speeding and we were before, clear ALL alert bits
    ; in flag1 and stop alert ( this ensures any other alert, eg camera, can
    ; restart
    
    movlw   LOW SPEED_70MPH
    subwf   SpeedL, W
    movlw   HIGH SPEED_70MPH
    subwfb  SpeedH, W
    btfss   STATUS, C 
    goto    AD70L000
    
    ; exceeding the speed.
    ; Are we already exceeding the speed? Goto AD70L002 if so
    ; and start alert if not
    btfsc   flags1, OVERSPEED_FLAG1
    goto    AD70L002
    
    bsf     flags1, OVERSPEED_FLAG1
    movlw   ALERT_OVER_SPEED
    movwf   tuneCurrent
    goto    AD70L002
    
AD70L000    
    ; Not exceeding the speed
    ; were we exceeding the speed? Goto AD70L002 if not
    ; and start alert if not
    btfss   flags1, OVERSPEED_FLAG1
    goto    AD70L002
    
    bcf     flags1, OVERSPEED_FLAG1
    movlw   ALERT_SILENCE
    movwf   tuneCurrent
    bcf     flags1, MASK_ALERT_FLAG1
   
AD70L002
    call    DspHome

    DISPLAY STR70MPH

    call    DspLine2   
    
    ; Do we have speed data yet? Warn user if not
    movf    SpeedH, W
    iorwf   SpeedL, W 
    btfss   STATUS, Z
    bra     AD70L003
    
    DISPLAY STRNoSpeed
    return

AD70L003
    btfss   flags2, INLOCKFLAG2
    goto    AD70L001
    
    ; Display the distance as a bar graph
    call    DspBarGraph
        
    return

AD70L001    
    ; Just tell the user there is no signal
    DISPLAY STRNoLock

    return        







;*****************************************************************************        
;
;   Function :  AppDsp30KLimit
;               Displays the current closest camera as a bargraph and
;               also signals when the speed exceeds 30KM/H
;
;
;   Input:      Nothing specific
;
;   Output:     N/A
;
;*****************************************************************************        
AppDsp30KLimit
    tstfsz  appDspTmr
    return

    ; Restart the display update timer    
    movlw   D'10'
    movwf   appDspTmr

    ; If we are moving faster than 30, start to play the warning alert.
    ; Otherwise, turn it off.
    
    
    ; The test: current speed - 30
    ; if result is postive, ie carry is set we are not speed
    ;
    ; If we are speeding and we were not before, then set flags1 bit 
    ; and start alert.
    ; If we are not speeding and we were before, clear ALL alert bits
    ; in flag1 and stop alert ( this ensures any other alert, eg camera, can
    ; restart
    
    movlw   LOW SPEED_30KPH
    subwf   SpeedL, W
    movlw   HIGH SPEED_30KPH
    subwfb  SpeedH, W
    btfss   STATUS, C 
    goto    AD30KL000
    
    ; exceeding the speed.
    ; Are we already exceeding the speed? Goto AD30L002 if so
    ; and start alert if not
    btfsc   flags1, OVERSPEED_FLAG1
    goto    AD30KL002
    
    bsf     flags1, OVERSPEED_FLAG1
    movlw   ALERT_OVER_SPEED
    movwf   tuneCurrent
    goto    AD30KL002
    
AD30KL000    
    ; Not exceeding the speed
    ; were we exceeding the speed? Goto AD30KL002 if not
    ; and start alert if not
    btfss   flags1, OVERSPEED_FLAG1
    goto    AD30KL002
    
    bcf     flags1, OVERSPEED_FLAG1
    movlw   ALERT_SILENCE
    movwf   tuneCurrent
    bcf     flags1, MASK_ALERT_FLAG1
   
AD30KL002
    call    DspHome

    DISPLAY STR30KPH

    call    DspLine2   
    
    ; Do we have speed data yet? Warn user if not
    movf    SpeedH, W
    iorwf   SpeedL, W 
    btfss   STATUS, Z
    bra     AD30KL003
    
    DISPLAY STRNoSpeed
    return

AD30KL003
    btfss   flags2, INLOCKFLAG2
    goto    AD30KL001
    
    ; Display the distance as a bar graph
    call    DspBarGraph
        
    return
 
AD30KL001    
    ; Just tell the user there is no signal
    DISPLAY STRNoLock

    return        
    
    

;*****************************************************************************        
;
;   Function :  AppDsp50KLimit
;               Displays the current closest camera as a bargraph and
;               also signals when the speed exceeds 50KM/H
;
;
;   Input:      Nothing specific
;
;   Output:     N/A
;
;*****************************************************************************        
AppDsp50KLimit
    tstfsz  appDspTmr
    return

    ; Restart the display update timer    
    movlw   D'10'
    movwf   appDspTmr

    ; If we are moving faster than 50, start to play the warning alert.
    ; Otherwise, turn it off.
    
    
    ; The test: current speed - 50
    ; if result is postive, ie carry is set we are not speed
    ;
    ; If we are speeding and we were not before, then set flags1 bit 
    ; and start alert.
    ; If we are not speeding and we were before, clear ALL alert bits
    ; in flag1 and stop alert ( this ensures any other alert, eg camera, can
    ; restart
    
    movlw   LOW SPEED_50KPH
    subwf   SpeedL, W
    movlw   HIGH SPEED_50KPH
    subwfb  SpeedH, W
    btfss   STATUS, C 
    goto    AD50KL000
    
    ; exceeding the speed.
    ; Are we already exceeding the speed? Goto AD50L002 if so
    ; and start alert if not
    btfsc   flags1, OVERSPEED_FLAG1
    goto    AD50KL002
    
    bsf     flags1, OVERSPEED_FLAG1
    movlw   ALERT_OVER_SPEED
    movwf   tuneCurrent
    goto    AD50KL002
    
AD50KL000    
    ; Not exceeding the speed
    ; were we exceeding the speed? Goto AD50KL002 if not
    ; and start alert if not
    btfss   flags1, OVERSPEED_FLAG1
    goto    AD50KL002
    
    bcf     flags1, OVERSPEED_FLAG1
    movlw   ALERT_SILENCE
    movwf   tuneCurrent
    bcf     flags1, MASK_ALERT_FLAG1
   
AD50KL002
    call    DspHome

    DISPLAY STR50KPH

    call    DspLine2   
    
    ; Do we have speed data yet? Warn user if not
    movf    SpeedH, W
    iorwf   SpeedL, W 
    btfss   STATUS, Z
    bra     AD50KL003
    
    DISPLAY STRNoSpeed
    return

AD50KL003
    btfss   flags2, INLOCKFLAG2
    goto    AD50KL001
    
    ; Display the distance as a bar graph
    call    DspBarGraph
        
    return

AD50KL001    
    ; Just tell the user there is no signal
    DISPLAY STRNoLock

    return            


;*****************************************************************************        
;
;   Function :  AppDsp80KLimit
;               Displays the current closest camera as a bargraph and
;               also signals when the speed exceeds 80KM/H
;
;
;   Input:      Nothing specific
;
;   Output:     N/A
;
;*****************************************************************************        
AppDsp80KLimit
    tstfsz  appDspTmr
    return

    ; Restart the display update timer    
    movlw   D'10'
    movwf   appDspTmr

    ; If we are moving faster than 80, start to play the warning alert.
    ; Otherwise, turn it off.
    
    
    ; The test: current speed - 80
    ; if result is postive, ie carry is set we are not speed
    ;
    ; If we are speeding and we were not before, then set flags1 bit 
    ; and start alert.
    ; If we are not speeding and we were before, clear ALL alert bits
    ; in flag1 and stop alert ( this ensures any other alert, eg camera, can
    ; restart
    
    movlw   LOW SPEED_80KPH
    subwf   SpeedL, W
    movlw   HIGH SPEED_80KPH
    subwfb  SpeedH, W
    btfss   STATUS, C 
    goto    AD80KL000
    
    ; exceeding the speed.
    ; Are we already exceeding the speed? Goto AD80L002 if so
    ; and start alert if not
    btfsc   flags1, OVERSPEED_FLAG1
    goto    AD80KL002
    
    bsf     flags1, OVERSPEED_FLAG1
    movlw   ALERT_OVER_SPEED
    movwf   tuneCurrent
    goto    AD80KL002
    
AD80KL000    
    ; Not exceeding the speed
    ; were we exceeding the speed? Goto AD80KL002 if not
    ; and start alert if not
    btfss   flags1, OVERSPEED_FLAG1
    goto    AD80KL002
    
    bcf     flags1, OVERSPEED_FLAG1
    movlw   ALERT_SILENCE
    movwf   tuneCurrent
    bcf     flags1, MASK_ALERT_FLAG1
   
AD80KL002
    call    DspHome

    DISPLAY STR80KPH

    call    DspLine2   
    
    ; Do we have speed data yet? Warn user if not
    movf    SpeedH, W
    iorwf   SpeedL, W 
    btfss   STATUS, Z
    bra     AD80KL003
    
    DISPLAY STRNoSpeed
    return

AD80KL003
    btfss   flags2, INLOCKFLAG2
    goto    AD80KL001
    
    ; Display the distance as a bar graph
    call    DspBarGraph
        
    return

AD80KL001    
    ; Just tell the user there is no signal
    DISPLAY STRNoLock

    return          




;*****************************************************************************        
;
;   Function :  AppDsp100KLimit
;               Displays the current closest camera as a bargraph and
;               also signals when the speed exceeds 100KM/H
;
;
;   Input:      Nothing specific
;
;   Output:     N/A
;
;*****************************************************************************        
AppDsp100KLimit
    tstfsz  appDspTmr
    return

    ; Restart the display update timer    
    movlw   D'10'
    movwf   appDspTmr

    ; If we are moving faster than 100, start to play the warning alert.
    ; Otherwise, turn it off.
    
    
    ; The test: current speed - 100
    ; if result is postive, ie carry is set we are not speed
    ;
    ; If we are speeding and we were not before, then set flags1 bit 
    ; and start alert.
    ; If we are not speeding and we were before, clear ALL alert bits
    ; in flag1 and stop alert ( this ensures any other alert, eg camera, can
    ; restart
    
    movlw   LOW SPEED_100KPH
    subwf   SpeedL, W
    movlw   HIGH SPEED_100KPH
    subwfb  SpeedH, W
    btfss   STATUS, C 
    goto    AD100KL000
    
    ; exceeding the speed.
    ; Are we already exceeding the speed? Goto AD100L002 if so
    ; and start alert if not
    btfsc   flags1, OVERSPEED_FLAG1
    goto    AD100KL002
    
    bsf     flags1, OVERSPEED_FLAG1
    movlw   ALERT_OVER_SPEED
    movwf   tuneCurrent
    goto    AD100KL002
    
AD100KL000    
    ; Not exceeding the speed
    ; were we exceeding the speed? Goto AD100KL002 if not
    ; and start alert if not
    btfss   flags1, OVERSPEED_FLAG1
    goto    AD100KL002
    
    bcf     flags1, OVERSPEED_FLAG1
    movlw   ALERT_SILENCE
    movwf   tuneCurrent
    bcf     flags1, MASK_ALERT_FLAG1
   
AD100KL002
    call    DspHome

    DISPLAY STR100KPH

    call    DspLine2   
    
    ; Do we have speed data yet? Warn user if not
    movf    SpeedH, W
    iorwf   SpeedL, W 
    btfss   STATUS, Z
    bra     AD100KL003
    
    DISPLAY STRNoSpeed
    return

AD100KL003
    btfss   flags2, INLOCKFLAG2
    goto    AD100KL001
    
    ; Display the distance as a bar graph
    call    DspBarGraph
        
    return

AD100KL001    
    ; Just tell the user there is no signal
    DISPLAY STRNoLock

    return          



;*****************************************************************************        
;
;   Function :  AppDsp120KLimit
;               Displays the current closest camera as a bargraph and
;               also signals when the speed exceeds 120KM/H
;
;
;   Input:      Nothing specific
;
;   Output:     N/A
;
;*****************************************************************************        
AppDsp120KLimit
    tstfsz  appDspTmr
    return

    ; Restart the display update timer    
    movlw   D'10'
    movwf   appDspTmr

    ; If we are moving faster than 120, start to play the warning alert.
    ; Otherwise, turn it off.
    
    
    ; The test: current speed - 120
    ; if result is postive, ie carry is set we are not speed
    ;
    ; If we are speeding and we were not before, then set flags1 bit 
    ; and start alert.
    ; If we are not speeding and we were before, clear ALL alert bits
    ; in flag1 and stop alert ( this ensures any other alert, eg camera, can
    ; restart
    
    movlw   LOW SPEED_120KPH
    subwf   SpeedL, W
    movlw   HIGH SPEED_120KPH
    subwfb  SpeedH, W
    btfss   STATUS, C 
    goto    AD120KL000
    
    ; exceeding the speed.
    ; Are we already exceeding the speed? Goto AD120L002 if so
    ; and start alert if not
    btfsc   flags1, OVERSPEED_FLAG1
    goto    AD120KL002
    
    bsf     flags1, OVERSPEED_FLAG1
    movlw   ALERT_OVER_SPEED
    movwf   tuneCurrent
    goto    AD120KL002
    
AD120KL000    
    ; Not exceeding the speed
    ; were we exceeding the speed? Goto AD120KL002 if not
    ; and start alert if not
    btfss   flags1, OVERSPEED_FLAG1
    goto    AD120KL002
    
    bcf     flags1, OVERSPEED_FLAG1
    movlw   ALERT_SILENCE
    movwf   tuneCurrent
    bcf     flags1, MASK_ALERT_FLAG1
   
AD120KL002
    call    DspHome

    DISPLAY STR120KPH

    call    DspLine2   
    
    ; Do we have speed data yet? Warn user if not
    movf    SpeedH, W
    iorwf   SpeedL, W 
    btfss   STATUS, Z
    bra     AD120KL003
    
    DISPLAY STRNoSpeed
    return

AD120KL003
    btfss   flags2, INLOCKFLAG2
    goto    AD120KL001
    
    ; Display the distance as a bar graph
    call    DspBarGraph
        
    return

AD120KL001    
    ; Just tell the user there is no signal
    DISPLAY STRNoLock

    return          


; Reuse some of the scan loop variables; they are free now
testDistH   EQU AARGB7      ; Current distance from table
testDistL   EQU AARGB6
tmpDist     EQU AARGB5      ; stores result of calculation 
barChar     EQU AARGB4      ; The character that will be displayed in the bargraph
barLoop     EQU AARGB3      ; Count the # of bars drawn ( up to 70 )
barIndex    EQU AARGB2      ; Count of 5, for each character cell
tmpAlert    EQU AARGB2      ; Holds the audible alert number to generate, if any

;*****************************************************************************        
;
;   Function :  DspBarGraph
;               Writes the bar graph line onto the display, using scanDist 
;               as the distance away.
;               It also checks the distance against the 3 audible warning 
;               points, generating the relevant alert if needed
;
;   Input:      scanDist, scanDist+1, scanDist+2
;
;   Output:     16 characters written to the display
;               The caller should have positioned the cursor at the start
;               of line 2 before invoking this function.
;
;*****************************************************************************        
DspBarGraph
    ; Display graph or 'no cameras in range'?
    tstfsz  scanDist
    goto    DBG001
    
    movlw   0x80
    andwf   scanDist+1, W
    btfss   STATUS,Z
    goto    DBG001
        
    ; Before we do anything else, check the three preset alert distances, and sound the
    ; higest priority alert
    
    clrf    tmpAlert
    
    ; Actually before we do anything, see if the speed is less than the lowest limit. 
    ; Do not generate an alert in that case
    

    ; Do we have valid speed data?
    ; Do we have any valid speed data? 
    btfss   flags1, SPEEDHEAD_VALID_FLAG1
    bra     DBG007    
    
    movlw   LOW SPEED_NO_ALERT
    subwf   SpeedL, W
    movlw   HIGH SPEED_NO_ALERT
    subwfb  SpeedH, W
    btfss   STATUS, C 
    goto    DBG004
    

    ; The test: scanDist - ALERT3_DISTANCE
    ; Carry is clear if ALERT3_DISTANCE > scanDist
DBG007    
    movlw   LOW ALERT3_DISTANCE
    subwf   scanDist+2, W
    movlw   HIGH ALERT3_DISTANCE
    subwfb  scanDist+1, W

    movlw   ALERT_CAMERA_HERE
    btfss   STATUS, C
    movwf   tmpAlert
    btfss   STATUS, C
    bra     DBG004

    movlw   LOW ALERT2_DISTANCE
    subwf   scanDist+2, W
    movlw   HIGH ALERT2_DISTANCE
    subwfb  scanDist+1, W

    movlw   ALERT_CAMERA_CLOSE
    btfss   STATUS, C
    movwf   tmpAlert
    btfss   STATUS, C
    bra     DBG004

    movlw   LOW ALERT1_DISTANCE
    subwf   scanDist+2, W
    movlw   HIGH ALERT1_DISTANCE
    subwfb  scanDist+1, W

    movlw   ALERT_CAMERA_NEAR
    btfss   STATUS, C
    movwf   tmpAlert 
    
DBG004
    ; We have the alert number in tmpAlert.
    ; Only write a new value if there is no other alert running
    btfsc   flags1, OVERSPEED_FLAG1
    bra     DBG003
    
    ; Do not write the 'no alert' signal if we are sounding a keypress
    movf    tmpAlert, W
    btfss   STATUS, Z
    bra     DBG006
    
    ; But you can cancel the alert suppression
    bcf     flags1, MASK_ALERT_FLAG1
    
    movlw   ALERT_KEY_PRESS
    subwf   tuneCurrent, W
    btfsc   STATUS,Z
    bra     DBG003
    bra     DBG005
    
DBG006
    ; dont write a new alert if the suppress flag is set
    btfsc   flags1, MASK_ALERT_FLAG1   
    bra     DBG003
        
DBG005        
    movf    tmpAlert, W
    movwf   tuneCurrent
    
DBG003    
    movlw   0x00
    call    DspPutChar

    movlw   D'14'       ; 14 character cells to fill
    movwf   barLoop
    
    ; Point to first, furthest distance value
    movlw   HIGH BarGraphPoints
    movwf   TBLPTRH
    movlw   LOW  BarGraphPoints
    movwf   TBLPTRL
    
DBG000
    ; Loop through the 14 characters     
    
    movlw   0x05
    movwf   barIndex
    
    clrf    barChar   

DBG002    
    ; Read the current evaluation distance
    tblrd*+
    movf    TABLAT, W
    movwf   testDistL    
    tblrd*+
    movf    TABLAT, W
    movwf   testDistH    
    
    ; The test: scanDist - testDist
    ;           if result is negative, update char with new, extra vertical bar 
    ;           else leave barChar unchanged

    movf    testDistL, W
    subwf   scanDist+2, W
    movf    testDistH, W
    subwfb  scanDist+1, W
    movwf   tmpDist
    
    btfsc   tmpDist, 7
    incf    barChar, F

    decfsz  barIndex, F
    bra     DBG002
    
    ; Display the cell
    movlw   ' '
    tstfsz  barChar
    movf    barChar, W    
    call    DspPutChar

    decfsz  barLoop, F
    goto    DBG000
    
    
    movlw   0x07
    call    DspPutChar

    return
    
DBG001
    ; No cameras in range
    
    DISPLAY STRNoCameras
    
    return        



;*****************************************************************************        
;
;   Function :  AppDspAltHeading
;               Display the Altitude and the heading
;
;
;   Input:      Nothing specific
;
;   Output:     N/A
;
;*****************************************************************************        
AppDspAltHeading
    tstfsz  appDspTmr
    return

    ; Restart the display update timer    
    movlw   D'10'
    movwf   appDspTmr

    call    DspHome
    call    DspClear

    DISPLAY STRAltitude

    ; Do we have any valid altitude data? ie, are we in lock?
    btfss   flags2, INLOCKFLAG2
    bra     ADAS003
    
    ; Dont display negative numbers, which can appear during signal acquisition
    movlw   '-'
    subwf   Alt1, W
    btfsc   STATUS,Z
    bra     ADAS003

    movf    Alt1, W
    btfss   STATUS, Z
    call    DspPutChar

    movf    Alt2, W
    btfss   STATUS, Z
    call    DspPutChar

    movf    Alt3, W
    btfss   STATUS, Z
    call    DspPutChar

    movf    Alt4, W
    btfss   STATUS, Z
    call    DspPutChar
    
    ; Do we actually have an altitute reading yet?
    movf    Alt1, W
    iorwf   Alt2, W
    iorwf   Alt3, W
    iorwf   Alt4, W

    btfss   STATUS, Z
    bra     ADAS001
        
ADAS003       
    ; No, so display it as such
    DISPLAY STRUnknown
    bra     ADAS002 

ADAS001
    movlw   'M'
    call    DspPutChar

ADAS002
    call    DspLine2    
    

    btfss   flags2, INLOCKFLAG2
    goto    ADAS004
    
    ; Display the heading 
    DISPLAY STRHeading
    
      
      

    movf    Head1, W, BANKED
    btfss   STATUS, Z
    call    DspPutChar

    movf    Head2, W, BANKED
    btfss   STATUS, Z
    call    DspPutChar

    movf    Head3, W, BANKED
    btfss   STATUS, Z
    call    DspPutChar
    
    ; Do we actually have a heading reading yet?
    movf    Head1, W, BANKED
    iorwf   Head2, W, BANKED
    iorwf   Head3, W, BANKED

    btfss   STATUS, Z
    goto    ADAS005
        
    ; No, so display it as such
    DISPLAY STRUnknown
    return

ADAS004  
    ; Just tell the user there is no signal
    DISPLAY STRNoLock
    return        
    
ADAS005
    movlw   0x06
    call    DspPutChar
    return    
    
    
    
    







;*****************************************************************************        
;
;   Function :  AppDspSpeed
;               Display the current speed in Miles/h and KM/h
;
;
;   Input:      Nothing specific
;
;   Output:     N/A
;
;*****************************************************************************        
AppDspSpeed
    tstfsz  appDspTmr
    return

    ; Restart the display update timer    
    movlw   D'10'
    movwf   appDspTmr

    call    DspHome
    call    DspClear


    ; are we in lock?
    btfss   flags2, INLOCKFLAG2
    bra     ADSS001

    ; Do we have any valid speed data? 
    btfss   flags1, SPEEDHEAD_VALID_FLAG1
    bra     ADSS002
    
    movlw   ' '
    call    DspPutChar
    movlw   ' '
    call    DspPutChar
    
    ; Convert the speed to MPH.
    ; I make that a conversion factor of about 1.2, ie 19/16
    ; That is a nice easy calculation
    movf    SpeedL, W
    mullw   D'19'
    rrcf    PRODH, F
    rrcf    PRODL, F
    rrcf    PRODH, F
    rrcf    PRODL, F
    rrcf    PRODH, F
    rrcf    PRODL, F
    rrcf    PRODH, F
    rrcf    PRODL, W
    movwf   bin
    
    call    bin2BCD

    movf    hundreds, W
    addlw   '0'
    movf    hundreds, F     ; Test file for zero
    btfsc   STATUS, Z
    movlw   ' '    
    call    DspPutChar

    ; Dont display tens if zero and if hundreds is zero
    swapf   tensones, W
    andlw   0x0F
    iorwf   hundreds, F     ; A very cunny way to combine the test
    addlw   '0'
    
    movf    hundreds, F     ; Test file for zero
    btfss   STATUS, Z
    goto    ADSS003
    
    movlw   ' '
    
ADSS003
    call    DspPutChar

    movf    tensones, W
    andlw   0x0F
    addlw   '0'
    call    DspPutChar

    movlw   ' '
    call    DspPutChar
    movlw   'M'
    call    DspPutChar
    movlw   'P'
    call    DspPutChar
    movlw   'H'
    call    DspPutChar


    call    DspLine2  
      
    
    movlw   ' '
    call    DspPutChar
    movlw   ' '
    call    DspPutChar
    
    ; Convert the speed to KMH.
    ; I make that a conversion factor of about 1.91, ie 61/32
    ; That is a nice easy calculation
    movf    SpeedL, W
    mullw   D'61'
    rrcf    PRODH, F
    rrcf    PRODL, F
    rrcf    PRODH, F
    rrcf    PRODL, F
    rrcf    PRODH, F
    rrcf    PRODL, F
    rrcf    PRODH, F
    rrcf    PRODL, F
    rrcf    PRODH, F
    rrcf    PRODL, W
    movwf   bin

    call    bin2BCD
    
    movf    hundreds, W
    addlw   '0'
    movf    hundreds, F     ; Test file for zero
    btfsc   STATUS, Z
    movlw   ' '    
    call    DspPutChar

    ; Dont display tens if zero and if hundreds is zero
    swapf   tensones, W
    andlw   0x0F
    iorwf   hundreds, F     ; A very cunny way to combine the test
    addlw   '0'
    
    movf    hundreds, F     ; Test file for zero
    btfss   STATUS, Z
    goto    ADSS004
    
    movlw   ' '
    
ADSS004
    call    DspPutChar

    movf    tensones, W
    andlw   0x0F
    addlw   '0'
    call    DspPutChar

    movlw   ' '
    call    DspPutChar
    movlw   'K'
    call    DspPutChar
    movlw   'M'
    call    DspPutChar
    movlw   'H'
    call    DspPutChar
    
    return 

ADSS001  
    ; Just tell the user there is no signal
    DISPLAY STRNoLock
    return        

ADSS002
    ; Just tell the user there is speed data
    DISPLAY STRNoSpeed
    return        
    

; Translates between scandist and real distance in 43M intervals ending at 0
; 1 Interval = 1 vertical bar on the BarGraph
BarGraphPoints  
    DW 0x63E6,0x6106,0x5E31,0x5B67,0x58A7
    DW 0x55F2,0x5348,0x50A8,0x4E14,0x4B8A
    DW 0x490A,0x4696,0x442C,0x41CD,0x3F79
    DW 0x3D30,0x3AF1,0x38BD,0x3693,0x3475
    DW 0x3261,0x3058,0x2E5A,0x2C66,0x2A7D
    DW 0x289F,0x26CC,0x2503,0x2346,0x2193
    DW 0x1FEA,0x1E4D,0x1CBA,0x1B32,0x19B4
    DW 0x1842,0x16DA,0x157C,0x142A,0x12E2
    DW 0x11A5,0x1073,0x0F4C,0x0E2F,0x0D1D
    DW 0x0C16,0x0B1A,0x0A28,0x0941,0x0865
    DW 0x0793,0x06CC,0x0610,0x055F,0x04B9
    DW 0x041D,0x038C,0x0306,0x028A,0x0219
    DW 0x01B3,0x0158,0x0107,0x00C1,0x0086
    DW 0x0056,0x0030,0x0015,0x0005,0x0001
    
    
;*****************************************************************************
;
; Function : bin2BCD
;            Converts an 8 bit number in 'bin' into a packed BCD number in
;            'hundreds' and tensones
;
; Input: None.
;
; Output: N/A
;
;*****************************************************************************    
bin2BCD
    clrf    hundreds
    swapf   bin, W
    addwf   bin, W
    andlw   0x0F
    btfsc   STATUS,DC
    addlw   0x16
    btfsc   STATUS,DC
    addlw   0x06
    addlw   0x06
    btfss   STATUS,DC
    addlw   0xFA
    
    btfsc   bin,4
    addlw   0x1B
    btfss   STATUS,DC 
    addlw   0xFA
    
    btfsc   bin, 5
    addlw   0x30
    
    btfsc   bin, 6
    addlw   0x60
    
    btfsc   bin, 7
    addlw   0x20
    
    addlw   0x60
    rlcf    hundreds,F
    btfss   hundreds, 0
    addlw   0xA0
    
    movwf   tensones
    btfsc   bin,7
    incf    hundreds,F
    
    return    