;*****************************************************************************        
;
;   Module:     CA.ASM
;               
;   Author:     Mike Hibbett 
;                                                                  
;   Version:    1.1 20/11/04                                                  
;
;               Main source file for the EPE crossword assistant project
;               This file includes all other source files.
;
;*****************************************************************************        

    list p=16f877, st=OFF, x=OFF, n=0
    errorlevel -302
    errorlevel -306

    #include <p16f877.inc>

    __config  0x3ffa

    ; Pull in variable definitions and general constants
    #include "EQUATES.INC"

    EXPAND

;*****************************************************************************        
;
;   Function :  Reset vector
;               Hardware entry point to the code
;               This vector is also used for soft resets in the event of 
;               abnormal events
;
;   Input:      None.
;
;   Output:     N/A
;
;*****************************************************************************        
    
    ORG    0

RESET        GOTO    Main



;*****************************************************************************        
;
;   Function :  clearMemory
;               Zeros all memory ( bank 0 )
;               
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
clearMemory
    ; Clear all bank0 ram
    movlw   0x20
    movwf   FSR
clrMem
    clrf    INDF
    incf    FSR, F
    btfss   FSR, 7              ; Have we reached 0x80?
    goto    clrMem              ; - No, so continue with clear
    return



;*****************************************************************************        
;
;   Function :  checkKey
;               Sets up the keyboard, and looks to see if a key is pressed
;               
;
;   Input:      None
;
;   Output:     Z flag = 1 for no keypressed
;
;*****************************************************************************        
checkKey
    ; Do some basic h/w initialisation to get the system running
    ; TRIS C : 7 6 5 4 3 2 1 0
    ;          i i i i i i i i all inputs ( for key detection )
    ; TRIS A : 7 6 5 4 3 2 1 0
    ;          - - o i o o o o
    ; PORT A : 7 6 5 4 3 2 1 0
    ;          - - 1 0 1 0 1 1  ( LCD off, ROM off, keys enabled )
    ; TRIS B,D,E : outputs, for flash address bus
    
    movlw   0x2D
    movwf   PORTA
    bsf     STATUS, RP0     ; Select bank 1
    movlw   0x10
    movwf   TRISA
    movlw   0xFF
    movwf   TRISC
    movlw   0x000           ; Set outputs
    movwf   TRISB   
    movwf   TRISD
    movwf   TRISE
    movlw   0x06
    movwf   ADCON1
    bcf     STATUS, RP0     ; Select bank 0
    
    ; Now check keys, after a little startup delay
    movlw   D'5'
    call    uiWait10ms
    
    comf    DATA_BUS,W
    andlw   0x0F0
    
    
    btfsc   STATUS,Z
    return
    
    ; - Debounce
    ; Now check keys, after a little startup delay
    movlw   D'5'
    call    uiWait10ms
    
    comf    DATA_BUS,W
    andlw   0x0F0
    
    return
    
    
    
;*****************************************************************************        
;
;   Function :  initHardware
;               reconfigures the peripherals of the chip.
;               
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
initHardware
    ; Set up initial port directions:
    ; PORTA : RA0..RA5 OUTPUT
    ;         RA0..2,5 low, RA3,RA4 high
    
    
    ; PORTB : All output  ( ROM Address A0..A7 )
    ; PORTC : All output  ( ROM/KEYBOARD/LCD data )
    ; PORTD : All output  ( ROM Address A8..A15 )
    ; PORTE : All output  ( ROM Address A16..A18 )
    
    movlw   0x018
    movwf   lcdCtrl
    movwf   PORTA           ; LCD off, PROM off
    clrf    PORTC
    
    bsf     STATUS, RP0     ; Select bank 1
    movlw   0x0C0           ; RA5-0 outputs
    movwf   TRISA
    movlw   0x000           ; Set outputs
    movwf   TRISB   
    movwf   TRISC
    movwf   TRISD
    movwf   TRISE
    movlw   0x06
    movwf   ADCON1
    bcf     STATUS, RP0     ; Select bank 0
    return
    

;*****************************************************************************        
;
;   Function :  initDisplay
;               Setup the lcd display, 4bit interface
;               
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
initDisplay
    ; Display needs 14ms following power up - give it 50ms to be sure
    movlw   D'5'
    call    uiWait10ms
    
    movlw   0x28
    call    dspPutCmd

    movlw   0x28
    call    dspPutCmd

    movlw   0x28
    call    dspPutCmd

    movlw   0x28
    call    dspPutCmd

    movlw   0x00C
    call    dspPutCmd       ; disp on, no cursor            

    return
    


;*****************************************************************************        
;
;   Macro :  GETBYTE
;            Reads the contents of the PROM at the current memory location   
;            PROM is assumed enabled, but OE dis-asserted
;   Input:   
;
;   Output:  Data byte in W
;
;*****************************************************************************        
GETBYTE     macro   
    bcf     CONTROL, ROM_OE
    movfw   DATA_BUS
    bsf     CONTROL, ROM_OE
            endm



;*****************************************************************************        
;
;   Macro : INCPINS 
;           Increments the PROM address lines
;
;   Input:      
;
;   Output:     
;
;*****************************************************************************        
INCPINS     macro   
    incfsz  PROM_ADD_L,F
    goto    $+4
    incfsz  PROM_ADD_M,F
    goto    $+2
    incf    PROM_ADD_H,F
            endm



;*****************************************************************************        
;
;   Macro : ADDTOPINS  
;           Increments the PROM address lines by the value in W
;
;   Input:  Increment in W
;
;   Output:     
;
;*****************************************************************************        
ADDTOPINS   macro
    addwf   PROM_ADD_L,F
    btfss   STATUS,C
    goto    $+4
    incfsz  PROM_ADD_M,F
    goto    $+2
    incf    PROM_ADD_H,F    
            endm
            
            

;*****************************************************************************        
;
;   Macro : SUBFROMPINS 
;           Decrements the PROM address pins by the value in W  
;
;   Input:  Decrement in W
;
;   Output:     
;
;*****************************************************************************        
SUBFROMPINS macro
    subwf   PROM_ADD_L,F
    btfsc   STATUS,C
    goto    $+5
    decf    PROM_ADD_M,F
    incfsz   PROM_ADD_M,W
    goto    $+2
    decf    PROM_ADD_H,F
            endm
               
            
            
;*****************************************************************************        
;
;   Function :  Main
;               Main application loop
;
;   Input:      None.
;
;   Output:     N/A
;
;*****************************************************************************        
Main
    clrf    STATUS                ; Select bank 0
    clrf    INTCON                ; No interrupts
    clrf    PCLATH                ; Code is in first bank

    call    clearMemory
    call    checkKey                ; returns Z=1 if key pressed
    btfss   STATUS,Z
    call    updateFlash            ; This never returns    
    
    call    initHardware  
    call    initDisplay
    
    call    dspHome
    call    dspClear
    
    movlw   WELCOME_L1
    call    uiDspStr1
    call    dspLine2    
    movlw   WELCOME_L2
    call    uiDspStr1
    
    movlw   D'200'
    call    uiWait10ms
    
dspOptions    
    ; Ask user for option: Word or Anagram
    call    selectOption
    
    ; Jump to the appropriate option
    btfsc   optionMode,0
    goto    anagram
    
    ; Word search
    ; -----------
wl001    
    call    selectLength
    btfsc   STATUS,C
    goto    dspOptions

    call    dspClear    
    movlw   ENTER_LETTERS
    call    uiDspStr1
    movlw   D'100'
    call    uiWait10ms

wletters    
    call    enterLetters
    btfsc   STATUS,C
    goto    wl001

    call    dspClear    

    ; power up the prom
    bcf     CONTROL, ROM_CE

    ; Data bus must be an input    
    bsf     STATUS, RP0     ; Select bank 1
    movlw   0x0FF           ; Set inputs
    movwf   DATA_BUS_TRIS
    bcf     STATUS, RP0     ; Select bank 0

    ; Get the address of the first word of the specified address
    ; Z flag set if no words of that length preset
    call    find_start
    btfsc   STATUS,Z
    goto    no_words
    
loop
    call    fetch_word
    
    sublw   0x01            ; Are we at the end of the list?
    btfsc   STATUS,Z
    goto    no_more

    call    get_word
    
    call    display_word
    
    ; User presses a key to move to the next one    
    call    waitNoKey
wkey
    call    waitKey
    
    movwf   tmpVal
    btfsc   tmpVal, KEY_SHIFT_BIT 
    goto    wshift
    btfsc   tmpVal, KEY_DO_BIT
    goto    wnext 
    goto    wkey

wshift
    btfss   tmpVal, KEY_DO_BIT      
    goto    wkey
    goto    wletters
       
wnext
     call    dspClear    

    ; Data bus must be an input for PROM reading   
    bsf     STATUS, RP0     ; Select bank 1
    movlw   0x0FF           ; Set inputs
    movwf   DATA_BUS_TRIS
    bcf     STATUS, RP0     ; Select bank 0
    
    INCPINS                 ; Move to the next word
    
    goto    loop
    
no_words
    call    dspClear
    movlw   NO_WORDS_FOUND
    call    uiDspStr1

    movlw   D'100'
    call    uiWait10ms

    goto    dspOptions
    
no_more
    call    dspClear
    movlw   NO_MORE_WORDS
    call    uiDspStr1

    movlw   D'100'
    call    uiWait10ms

    goto    dspOptions
    
    
    ; Anagram search
    ; --------------
    
anagram
    call    selectLength
    btfsc   STATUS,C
    goto    dspOptions
    
    call    dspClear    
    movlw   ENTER_LETTERS
    call    uiDspStr1
    movlw   D'100'
    call    uiWait10ms

aletters    
    call    enterLetters
    btfsc   STATUS,C
    goto    anagram
    
    call    dspClear    

    bcf     CONTROL, ROM_CE

    ; Data bus must be an input    
    bsf     STATUS, RP0     ; Select bank 1
    movlw   0x0FF           ; Set inputs
    movwf   DATA_BUS_TRIS
    bcf     STATUS, RP0     ; Select bank 0

    call    find_start
    btfsc   STATUS,Z
    goto    no_words

aloop
    call    fetch_anagram
    
    sublw   0x01            ; Are we at the end of the list?
    btfsc   STATUS,Z
    goto    no_more

    call    get_word
    
    call    display_word
    
    ; User presses a key to move to the next one    
akey
    call    waitNoKey
    call    waitKey
    
    movwf   tmpVal
    btfsc   tmpVal, KEY_SHIFT_BIT 
    goto    ashift
    btfsc   tmpVal, KEY_DO_BIT
    goto    anext 
    goto    akey

ashift
    btfss   tmpVal, KEY_DO_BIT      
    goto    akey
    goto    aletters
       
anext
    call    dspClear    

    ; Data bus must be an input for PROM reading   
    bsf     STATUS, RP0     ; Select bank 1
    movlw   0x0FF           ; Set inputs
    movwf   DATA_BUS_TRIS
    bcf     STATUS, RP0     ; Select bank 0
    
    INCPINS                 ; Move to the next word

    goto    aloop    



;*****************************************************************************        
;
;   Function :  enterLetters
;               Allow the user to specify the known letters
;
;   Input:      
;
;   Output:     Result: dataBuffer filled
;               Carry set if ESC pressed
;
;*****************************************************************************        
enterLetters
    movlw   dataBuffer
    movwf   FSR
setBuf
    movlw   0x80
    movwf   INDF
    incf    FSR, F
    movfw   FSR
    sublw   dataBuffer+D'32'
    btfss   STATUS,Z
    goto    setBuf
    
    call    dspClear
    movlw   '>'
    call    dspPutChar

    clrf    position

el_000    
    movlw   '*'
    call    dspPutChar
    incf    position,F
    movlw   D'15'
    subwf   position,W
    btfss   STATUS,Z
    goto    el_001
    
    ; Move to the next line
    call    dspLine2
el_001
    movfw   inStrLen
    subwf   position,W
    btfss   STATUS,Z
    goto    el_000
    
    movlw   '<'
    call    dspPutChar
    
    ; Now display the cursor, allow it to move between the two lines,
    ; allow character to change.
    
    movlw   0x00D
    call    dspPutCmd       ; disp on, blinking cursor      

    clrf    position

el_002
    movlw   0x40 - D'16'
    movwf   tmpVal
    movlw   D'15'
    subwf   position,W
    movlw   1+0x80
    btfsc   STATUS,C
    addwf   tmpVal,W
    addwf   position,W
    call    dspPutCmd
    
    call    waitNoKey
el_002a    
    call    waitKey
    movwf   tmpVal
    btfsc   tmpVal, KEY_SHIFT_BIT
    goto    el_005
    btfsc   tmpVal, KEY_LEFT_BIT
    goto    el_003
    btfsc   tmpVal, KEY_RIGHT_BIT
    goto    el_004
    btfss   tmpVal, KEY_DO_BIT
    goto    el_002
    movlw   0x00C
    call    dspPutCmd       ; cursor off
    bcf     STATUS,C
    return

el_003
    decf    position,F
    movlw   0xff
    subwf   position,W
    btfsc   STATUS,Z
    incf    position,F
    goto    el_002
    
el_004
    incf    position,F    
    movfw   inStrLen
    subwf   position,W
    btfsc   STATUS,Z
    decf    position,F
    goto    el_002        
    
el_005
    btfss   tmpVal, KEY_DO_BIT
    goto    el_005b
    bsf     STATUS,C
    return
    
el_005b
    btfsc   tmpVal, KEY_LEFT_BIT
    goto    el_006
    btfss   tmpVal, KEY_RIGHT_BIT
    goto    el_002a        

    ; Wild card value is 0x80, displayed as *

    ; Right == up char
    movlw   dataBuffer
    addwf   position,W
    movwf   FSR
    incf    INDF,W    
    btfsc   INDF,7
    movlw   'A'
    movwf   INDF
    movlw   'Z' + 1
    subwf   INDF,W
    btfss   STATUS,Z
    goto    el_005a
    movlw   0x80
    movwf   INDF
    
el_005a    
    movfw   INDF
    btfsc   INDF,7
    movlw   '*'
    call    dspPutChar
    goto    el_002
    
el_006
    ; Left == down char
    movlw   dataBuffer
    addwf   position,W
    movwf   FSR
    decf    INDF,W    
    btfsc   INDF,7
    movlw   'Z'
    movwf   INDF
    movlw   'A' - 1
    subwf   INDF,W
    btfss   STATUS,Z
    goto    el_006a
    movlw   0x80
    movwf   INDF

el_006a
    movfw   INDF
    btfsc   INDF,7
    movlw   '*'
    call    dspPutChar
    goto    el_002


;*****************************************************************************        
;
;   Function :  selectOption
;               Asks the user to choose the operational option, Word or
;               Anagram.
;
;   Input:      
;
;   Output:     Result in option: 0 == Word, 0xff== Anagram
;
;*****************************************************************************        
selectOption
    clrf    optionMode
    call    dspClear 
    movlw   SELECT_OPTION
    call    uiDspStr1

so_002    
    call    dspLine2
    movlw   WORD_ANAGRAM1
    btfsc   optionMode, 0
    movlw   WORD_ANAGRAM2
    call    uiDspStr1

so_000    
    call    waitNoKey
    call    waitKey
    movwf   tmpVal
    btfsc   tmpVal, KEY_LEFT_BIT
    goto    so_001
    btfsc   tmpVal, KEY_RIGHT_BIT
    goto    so_001
    btfss   tmpVal, KEY_DO_BIT
    goto    so_000
    return

so_001
    comf    optionMode,F
    goto    so_002



;*****************************************************************************        
;
;   Function :  selectLength
;               Asks the user to choose the length of the word to search
;
;   Input:      
;
;   Output:     Result in inStrLen: 3 -- 30 characters
;               carry flag set if ESC pressed
;
;*****************************************************************************        
selectLength
    movlw   0x00C
    call    dspPutCmd       ; cursor off

    movlw   0x03
    movwf   inStrLen
    call    dspClear 
    movlw   SELECT_LENGTH
    call    uiDspStr1

sl_000    
    call    dspLine2
    
    ; Display length
    movlw   D'30'
    subwf   inStrLen,W
    btfss   STATUS,C
    goto    sl_001      ; next test
    movlw   '3'
    call    dspPutChar
    movlw   D'30'
    subwf   inStrLen,W  ; remove 10's digit
    goto    sl_digit
sl_001    
    movlw   D'20'
    subwf   inStrLen,W
    btfss   STATUS,C
    goto    sl_002      ; next test
    movlw   '2'
    call    dspPutChar
    movlw   D'20'
    subwf   inStrLen,W  ; remove 10's digit
    goto    sl_digit
sl_002
    movlw   D'10'
    subwf   inStrLen,W
    btfsc   STATUS,C
    goto    sl_002a
    movfw   inStrLen
    goto    sl_digit
    
sl_002a 
    movlw   '1'
    call    dspPutChar
    movlw   D'10'
    subwf   inStrLen,W  ; remove 10's digit

sl_digit
    addlw   '0'
    call    dspPutChar
    movlw   ' '
    call    dspPutChar

sl_003
    call    waitNoKey
sl_003a
    call    waitKey
    movwf   tmpVal
    btfsc   tmpVal, KEY_SHIFT_BIT
    goto    sl_004a
    btfsc   tmpVal, KEY_LEFT_BIT
    goto    sl_004
    btfsc   tmpVal, KEY_RIGHT_BIT
    goto    sl_005
    btfss   tmpVal, KEY_DO_BIT
    goto    sl_003
    bcf     STATUS,C
    return

sl_004a
    btfss   tmpVal, KEY_DO_BIT
    goto    sl_003a        
    bsf     STATUS,C
    return

sl_004
    decf    inStrLen,F
    movlw   D'2'
    subwf   inStrLen,W
    btfsc   STATUS,Z
    incf    inStrLen,F
    goto    sl_000

sl_005
    incf    inStrLen,F    
    movlw   D'31'
    subwf   inStrLen,W
    btfsc   STATUS,Z
    decf    inStrLen,F
    goto    sl_000


        
;*****************************************************************************        
;
;   Function :  getkey
;               does a raw read of the keyboard
;
;   Input:      
;
;   Output:     keys active in W
;
;*****************************************************************************        
getkey
    ; Data bus must be an input  
    bsf     CONTROL, KEY_EN 
    bsf     STATUS, RP0     ; Select bank 1
    movlw   0x0FF           ; Set inputs
    movwf   DATA_BUS_TRIS
    bcf     STATUS, RP0     ; Select bank 0

gk001
    comf    DATA_BUS,W
    bcf     CONTROL, KEY_EN 
    andlw   0x0F0
    return
    
    
    
;*****************************************************************************        
;
;   Function : waitNoKey 
;              Waits for all the keys to be released, with a debounce 
;
;   Input:      
;
;   Output:    W destroyed 
;
;*****************************************************************************        
waitNoKey
    call    getkey
    andlw   ~KEY_SHIFT
    btfss   STATUS,Z
    goto    waitNoKey
    movlw   D'25'
    call    uiWait10ms
    call    getkey
    andlw   ~KEY_SHIFT
    btfss   STATUS,Z
    goto    waitNoKey
    return    
 
 
        
;*****************************************************************************        
;
;   Function : waitKey 
;              waits for any key to be pressed, with no debounce 
;
;   Input:      
;
;   Output:    keys pressed in W
;
;*****************************************************************************        
waitKey
    call    getkey    
    btfsc   STATUS,Z
    goto    waitKey
    return
    


;*****************************************************************************        
;
;   Function : get_word  
;              Reads the most recent word found, and places it in a buffer 
;
;   Input:      
;
;   Output:    word in dspBuffer 
;
;*****************************************************************************        
get_word
    decf    inStrLen, W
    SUBFROMPINS
    movlw   dspBuffer
    movwf   FSR
    clrf    count

gw_001    
    GETBYTE
    movwf   INDF
    incf    FSR,F
    incf    count,F
    movfw   count
    subwf   inStrLen,W
    btfsc   STATUS,Z
    return

    INCPINS
    goto    gw_001
    
    

;*****************************************************************************        
;
;   Function : display_word 
;              copys the word in dspBuffer to the screen at the current 
;              cursor position  
;   Input:      
;
;   Output:    
;
;*****************************************************************************   
display_word
    movlw   dspBuffer
    movwf   FSR
    clrf    count

dw_001    
    movfw   INDF
    call    dspPutChar    
    incf    count,F
    incf    FSR,F
    movlw   D'17'
    subwf   count,W
    btfsc   STATUS,Z
    call    dspLine2
    movfw   count
    subwf   inStrLen,W
    btfss   STATUS,Z
    goto    dw_001
    return    



;*****************************************************************************        
;
;   Function : find_start 
;              Reads the index table to get the start address of the word list
;              and sets the PROM address lines to that value  
;   Input:     length of word in inStrLen
;
;   Output:    PROM pins modified
;              Z flag set if no words of that length exist in the PROM
;
;*****************************************************************************        
find_start
    movfw   inStrLen
    addwf   inStrLen,W
    movwf   tmpVal
    rlf     tmpVal,W
    movwf   PROM_ADD_L
    clrf    PROM_ADD_M
    clrf    PROM_ADD_H
    GETBYTE
    movwf   promAddH
    movwf   tmpVal
    incf    PROM_ADD_L,F
    GETBYTE
    movwf   promAddM
    iorwf   tmpVal,F
    incf    PROM_ADD_L,F
    GETBYTE
    movwf   promAddL
    iorwf   tmpVal,F
    movfw   promAddH
    movwf   PROM_ADD_H
    movfw   promAddM
    movwf   PROM_ADD_M
    movfw   promAddL
    movwf   PROM_ADD_L
    movfw   tmpVal
    return



;*****************************************************************************        
;
;   Function : fetch_anagram
;              scans the PROM from the current position until a matching word 
;              is found  
;   Input:     word template in dataBuffer
;
;   Output:    PROM pins modified, and point to the last character of the word
;              on exit
;              W holds status: 0x01 == no more words
;              0x00 == found one
;
;*****************************************************************************        
fetch_anagram
    clrf    count
    
fa_000
    movlw   dataBuffer
    movwf   FSR
    GETBYTE
    btfsc   STATUS, Z       ; If char is a 0, we are at the end of the list
    retlw   0x01            ; RETURN NO MORE    
    movwf   tmpVal
    
fa_001
    movfw   INDF
    xorwf   tmpVal,W
    btfsc   STATUS,Z    ; Have we found the byte in the original string?
    goto    nextchar    ; Yes - so mark it used, and move on
    
    incf    FSR,F
    
    ; Are we at the end of the original string?
    movfw   inStrLen
    addlw   dataBuffer
    subwf   FSR,W
    btfss   STATUS,Z
    goto    fa_001      ; No, so look at next char
    
    ; We failed to find character, so move to next word in the PROM
    movfw   count
    subwf   inStrLen,W
    ADDTOPINS
    
    movlw   dataBuffer
    movwf   FSR

    clrf    tmpVal
    
fa_003a    
    movlw   0x7F
    andwf   INDF,F
    incf    FSR, F
    incf    tmpVal,F
    movfw   inStrLen
    subwf   tmpVal,W
    btfss   STATUS,Z
    goto    fa_003a
    
    goto    fetch_anagram
    
nextchar
    ; Mark current letter as used
    movlw   0x80
    iorwf   INDF,F
    
    incf    count,F
    movfw   count
    subwf   inStrLen,W
    btfss   STATUS,Z
    goto    fa_002
    
    ; clear the MSB in the dataBuffer, since we will use it again later
    movlw   dataBuffer
    movwf   FSR

    clrf    tmpVal
    
fa_003    
    movlw   0x7F
    andwf   INDF,F
    incf    FSR, F
    incf    tmpVal,F
    movfw   inStrLen
    subwf   tmpVal,W
    btfss   STATUS,Z
    goto    fa_003
    
    retlw   0x00        ; We have matched all letters    
    
fa_002
    INCPINS
    goto    fa_000

    
;*****************************************************************************        
;
;   Function : fetch_word 
;              scans the PROM from the current position until a matching word 
;              is found  
;   Input:     word template in dataBuffer
;
;   Output:    PROM pins modified, and point to the last character of the word
;              on exit
;              W holds status: 0x01 == no more words
;              0x00 == found one
;
;*****************************************************************************  
fetch_word

    clrf    count
    movlw   dataBuffer
    movwf   FSR

fw_001
    GETBYTE
    btfsc   STATUS, Z       ; If first char is a 0, we are at the end of the list
    retlw   0x01            ; RETURN NO MORE    
    btfsc   INDF,7          ; msb set == wildcard, character unknown
    goto    fw_next
    xorwf   INDF,W
    btfss   STATUS,Z
    goto    fw_fail

fw_next    
    incf    FSR,F
    incf    count,F
    movfw   count
    subwf   inStrLen,W
    btfsc   STATUS,Z
    retlw   0
    
    INCPINS 
    goto    fw_001

fw_fail
    movfw   count
    subwf   inStrLen,W

    ADDTOPINS
    goto    fetch_word    
        


    #include "DISPLAY.INC"
    #include "FLASH.INC"


    END                ; End of program
    