;MOONCLOCK620.ASM 24MAR04 - Copyright John Becker - EPE PIC Moon & Tide Clock
;PIC16F877 3.2768MHz xtal, WDT off, POR on
; written in TASM for TK3 assembly

; NB Set calendar in order of year, month, day

                LIST P=16F877,R=DEC
                __config H'3F31' 

        include P16F877.inc

#DEFINE         BANK0 BCF 3,5           ; status bit 5
#DEFINE         BANK1 BSF 3,5           ; status bit 5
#DEFINE         AWRON movlw $B0         ; autowrite on command
#DEFINE         AWROFF movlw $B2        ; autowrite off command
#DEFINE         PEEK movlw $E0          ; screen peek command
#DEFINE         CSRPOS movlw $21        ; cursor position command
#DEFINE         OFFSET movlw $22        ; offset position command
#DEFINE         ADPSET movlw $24        ; set address pointer
#DEFINE         TXHOME movlw $40        ; text home address command
#DEFINE         TXAREA movlw $41        ; text area (columns) address command
#DEFINE         GRHOME movlw $42        ; graphics home address command
#DEFINE         GRAREA movlw $43        ; graphic area (columns) address command

        CBLOCK $20 ; start of automatic EQUates allocation, from $20 onwards

REGA0				;lsb
REGA1
REGA2
REGA3				;msb

REGB0				;lsb
REGB1
REGB2
REGB3				;msb

REGC0				;lsb
REGC1
REGC2
REGC3				;msb

DSIGN				;Digit Sign. 0=positive,FF(or non zero)=negative
DIGIT1				;MSD
DIGIT2
DIGIT3
DIGIT4
DIGIT5				;Decimal digits
DIGIT6
DIGIT7
DIGIT8
DIGIT9
DIGIT10				;LSD
MTEMP
MCOUNT
DCOUNT
LOOP

LOOPB                          ; general loop
LOOPC                          ; general loop
LOOPD                          ; general loop
LOOPE                          ; general loop
LOOPF                          ; general loop

STORE1                         ; temp store
STORE2                         ; temp store
TEMPA                          ; temp store
CLKCNT                         ; counter for pause val
ADRLSB                         ; low address
ADRMSB                         ; high address
ATTRIB                         ; attribute value
BITVAL                         ; val of bit to be set/reset
COLUMN                         ; column length holder
CLKSEC                         ; clock main counter - secs
CLKMIN                         ; clock - mins
CLKHRS                         ; clock - hours
DAY                            
MONTH                          
YEAR                           
MONTHLEN
WEEKDAY                        
CLKAD0                         ;clock rate adjustment lsb
CLKAD1                         ;clock rate adjustment nsb
CLKAD2                         ;clock rate adjustment msb
BYTE4                          
SECAD0         
SECAD1         
SECAD2         
TIDEBASIC
TIDEOFFSET
TIDESEGMENTMSB
TIDESEGMENTLSB
TIDECOUNTMSB
TIDECOUNTLSB
TIDEGRAPH
TIDEUP
MOONFACE
CHANGEVAL

BYTE
BYTE2
PAR
PAR2
STORE
SELECT

ANGLE
Y
X
B
MOONPHASE
MOONSTORE
LEFTMOON
MOONCOUNTMSB
MOONCOUNTLSB
NAMEVAL
KEYSTORE
TIDERESETMSB
TIDERESETLSB
MOONRESETMSB
MOONRESETLSB
BIRDFLAG

	ENDC

FS:             EQU %00000000          ;fs mode set by bit 5: 1 = 6x8, 0 = 8x8

                ORG 0
                goto 5                  
                ORG 4
                goto 5                  
                ORG 5
                goto START              

PLUSKEYTABLE:   movf CHANGEVAL,W
                addwf PCL,F
                return           ; 0
                goto INCMOON     ; 1
                goto INCTIDE     ; 2
                goto INCCALENDAR ; 3  weekday
                goto INCMONTH    ; 4
                goto INCCALENDAR ; 5  year
                goto INCCALENDAR ; 6  day
                goto INCCLOCK    ; 7  hours
                goto INCCLOCK    ; 8  mins
                goto INCADJUST   ; 9  clock accuracy adjust
                goto INCCLOCK    ; 10 secs
                goto INCTADJUST  ; 11 tide accuracy adjust
                goto INCMADJUST  ; 12 moon accuracy adjust

MINUSKEYTABLE:  movf CHANGEVAL,W
                addwf PCL,F
                return           ; 0
                goto DECMOON     ; 1
                goto DECTIDE     ; 2
                goto DECCALENDAR ; 3  weekday
                goto DECMONTH    ; 4
                goto DECCALENDAR ; 5  year
                goto DECCALENDAR ; 6  day
                goto DECCLOCK    ; 7  hours
                goto DECCLOCK    ; 8  mins
                goto DECADJUST   ; 9  clock accuracy adjust
                goto DECCLOCK    ; 10 secs
                goto DECTADJUST  ; 11 tide accuracy adjust
                goto DECMADJUST  ; 12 moon accuracy adjust

REGISTERTABLE:  movf CHANGEVAL,W
                addwf PCL,F
                return           ; 0
                retlw MOONFACE   ; 1
                retlw TIDEOFFSET ; 2
                retlw WEEKDAY    ; 3
                retlw MONTH      ; 4
                retlw YEAR       ; 5
                retlw DAY        ; 6
                retlw CLKHRS     ; 7
                retlw CLKMIN     ; 8
                retlw 0          ; 9 not used
                retlw CLKSEC     ; 10
                retlw TIDERESETMSB ; 11
                retlw MOONRESETMSB ; 12

LIMITTABLE:     movf CHANGEVAL,W
                addwf PCL,F
                return           ; 0
                retlw MOONFACE   ; 1
                retlw TIDEOFFSET ; 2
                retlw 6          ; 3   ; weekday
                retlw MONTH      ; 4
                retlw $99        ; 5   ; year
                goto LIMIT2      ; 6   ; day
                retlw $23        ; 7   ; hours
                retlw $59        ; 8   ; mins
                retlw 0          ; 9   ; not used
                retlw $59        ; 10  ; secs

LIMIT2:         movf MONTHLEN,W  ; 6   ; day
                return

SHOWINDEX:  movf SELECT,W
            xorlw 17
            btfsc STATUS,Z
            clrf SELECT
            incf SELECT,F
            movf SELECT,W
            addwf PCL,F
            nop                   ; 0
            goto SHOWCHECKMOON    ; 1
            goto SHOWCHECKTIDE    ; 2
            goto SHOWCHECKWEEKDAY ; 3
            goto SHOWCHECKMONTH   ; 4
            goto SHOWCHECKYEAR    ; 5
            goto SHOWCHECKDAY     ; 6
            goto SHOWCHECKHOUR    ; 7
            goto SHOWCHECKMINUTES ; 8
            goto SHOWTIMEADJUST   ; 9
            goto SHOWCHECKSECS    ; 10
            incf SELECT,F         ; 11
            incf SELECT,F         ; 12
            incf SELECT,F         ; 13
            incf SELECT,F         ; 14
            incf SELECT,F         ; 15
            goto SHOWTIDEADJUST   ; 16
            goto SHOWMOONADJUST   ; 17

SHOWINDEX2: addwf PCL,F
            nop                   ; 0
            goto SHOWCHECKMOON    ; 1
            goto SHOWCHECKTIDE    ; 2
            goto SHOWCHECKWEEKDAY ; 3
            goto SHOWCHECKMONTH   ; 4
            goto SHOWCHECKYEAR    ; 5
            goto SHOWCHECKDAY     ; 6
            goto SHOWCHECKHOUR    ; 7
            goto SHOWCHECKMINUTES ; 8
            goto SHOWTIMEADJUST   ; 9
            goto SHOWCHECKSECS    ; 10
            goto PLUSKEYTABLE     ; 11
            goto MINUSKEYTABLE    ; 12
            goto SHOWINDEX        ; 13
            goto ZEROSECS         ; 14
            goto SHOWEXIT         ; 15
            goto SHOWTIDEADJUST   ; 16
            goto SHOWMOONADJUST   ; 17
            goto TOGGLEBIRD       ; 18

NAMETABLE:      movf NAMEVAL,W
                addwf PCL,F
                retlw ' '       ; 0 moon
                retlw 'M'
                retlw 'O'
                retlw 'O'
                retlw 'N'
                retlw ' '
                retlw '#'

                retlw 'M'       ; 7 month
                retlw 'O'
                retlw 'N'
                retlw 'T'
                retlw 'H'
                retlw ' '
                retlw 'L'       ; Luna

                retlw ' '       ; 14 day
                retlw ' '
                retlw 'D'
                retlw 'A'
                retlw 'Y'
                retlw ' '
                retlw 'D'

                retlw ' '       ; 21 tide
                retlw 'T'
                retlw 'I'
                retlw 'D'
                retlw 'E'
                retlw ' '
                retlw 'T'

                retlw 'W'       ; 28 weekday
                retlw '-'
                retlw 'D'
                retlw 'A'
                retlw 'Y'
                retlw ' '
                retlw 'W'

                retlw ' '       ; 35 hours
                retlw 'H'
                retlw 'O'
                retlw 'U'
                retlw 'R'
                retlw ' '
                retlw 'H'

                retlw ' '       ; 42 year
                retlw 'Y'
                retlw 'E'
                retlw 'A'
                retlw 'R'
                retlw ' '
                retlw 'Y'

                retlw ' '       ; 49 mins
                retlw 'M'
                retlw 'I'
                retlw 'N'
                retlw 'S'
                retlw ' '
                retlw 'M'

                retlw 'T'       ; 56 clock timing adjust
                retlw '-'
                retlw 'A'
                retlw 'D'
                retlw 'J'
                retlw ' '
                retlw '%'

                retlw ' '       ; 63 secs
                retlw 'S'
                retlw 'E'
                retlw 'C'
                retlw 'S'
                retlw ' '
                retlw 'S'

                retlw 'C'       ; 70 Tide timing adjust
                retlw '-'
                retlw 'A'
                retlw 'D'
                retlw 'J'
                retlw ' '
                retlw 'C'       ; C for sea!

                retlw 'M'       ; 77 Moon timing adjust
                retlw '-'
                retlw 'A'
                retlw 'D'
                retlw 'J'
                retlw ' '
                retlw 'O'       ; O for Orbit

                retlw ' '

; **************** START OF SETTING UP **********

START:          bcf STATUS,RP0
                bcf STATUS,RP1          
                clrf PORTA              
                movlw %11111111         
                movwf PORTB             
                movlw %00001111         ; fs low, rst low, cd ce rd wr high
                movwf PORTC             
                clrf PORTD              
                clrf PORTE              
                BANK1                   
                movlw 255               
                movwf TRISA             ; porta as input
                movlw %10000110         ; 
                movwf ADCON1            ; set RA, RE digital

                movwf TRISB             ; porta as input
                movlw FS                
                movwf TRISC             ; portc as output graphic lcd control/fs
                clrf TRISD              ; portd as output graphic lcd d0-d7
                clrf TRISE              ; porte as output
                movlw %00000101         ; pull-up rs on (bit 7 lo), timer 1/50 sec
                movwf OPTION_REG        ; (for 3.2768mhz xtal)
                BANK0                   
                movlw %00011111         ; fs low,  rst cd ce rd wr high
                movwf PORTC             
                call PAUSIT             

                movlw 2
                movwf TIDESEGMENTMSB    ; segments of tide for graph display
                movlw $DC               ; (44714/94 = 475.681 = $01DC)
                movwf TIDESEGMENTLSB    ; see later
                movlw 93                ; 1 less than 94 because zero also counts
                movwf TIDEBASIC
                clrf TIDEOFFSET

                clrf KEYSTORE
                clrf MOONFACE
                clrf CHANGEVAL
                clrf SELECT
                movlw 1
                movwf BIRDFLAG

                movlw 25                ; initial basic clkcnt val for secs timing
                movwf CLKCNT            

GRAPHIC:        movlw 16                
                movwf COLUMN            ; set column length
                call SETUP              

                movlw %10000000         ; internal cg ram mode, or
                call SENDCMD            ; send command
                movlw %10011100         ; text & graphic on, cursor & blink off
                call SENDCMD            ; send command

                clrf PCLATH             

                movlw 0                 
                call PRMGET             
                movwf DAY               
                movlw 1                 
                call PRMGET             
                movwf MONTH             
                movlw 2                 
                call PRMGET             
                movwf YEAR              
                movlw 3                 
                call PRMGET             
                movwf WEEKDAY           
                movlw 4                 
                call PRMGET             
                movwf CLKAD0            
                movlw 5                 
                call PRMGET             
                movwf CLKAD1            
                movlw 6                 
                call PRMGET             
                movwf CLKAD2            
                movlw 7 
                call PRMGET             
                movwf CLKHRS            
                movlw 8                 
                call PRMGET             
                movwf CLKMIN            

                movlw 2
                movwf PCLATH
                bsf PCLATH,3
                call MONTH4             ; get days in month
                clrf PCLATH
                movwf MONTHLEN
                call LEAPYEAR

                clrf CLKSEC
                call CLKSHOW
                call DATESHOW

                movlw 13                
                call PRMGET             
                movwf TIDERESETMSB
                movwf TIDECOUNTMSB

                movlw 14
                call PRMGET             
                movwf TIDERESETLSB
                movwf TIDECOUNTLSB

                movlw 15                
                call PRMGET             
                movwf MOONRESETMSB
                movwf MOONCOUNTMSB

                movlw 16
                call PRMGET             
                movwf MOONRESETLSB
                movwf MOONCOUNTLSB

SETMOON:        movlw 11
                call PRMGET             
                movwf MOONFACE
                call MOONCALL

                movlw 47
                movwf TIDEGRAPH
SETTIDE:        call SHOWTIDE
                decfsz TIDEGRAPH,F
                goto SETTIDE

                movlw 10
                call PRMGET             
                movwf TIDEOFFSET
                movlw 12
                call PRMGET             
                movwf TIDEBASIC
                addwf TIDEOFFSET,W
                movwf TIDEGRAPH
                addlw 163           ; is greater than 93?
                btfsc STATUS,C
                movwf TIDEGRAPH     ; yes, so put new val into TIDEGRAPH
                call SHOWTIDE

                ;******** start of main program ***********

INTRPT:         comf PORTA,W            ; yes
                andlw %00000011         ; has a key been pressed?
                btfss STATUS,Z
                call KEYBOARD           ; yes

                btfss INTCON,2          ; has a timer time-out been detected?
                goto INTRPT             ; no

                bcf INTCON,2            ;
                call CLKADD             ; update time
                goto INTRPT

CLKADD:         decfsz CLKCNT,F         ; dec 1/25 clock counter. is it = 0?
                return                  ; no

                movlw 25                ; reset start value of clkcnt
                movwf CLKCNT            
                movf KEYSTORE,W
                btfss STATUS,Z
                call SHOWINDEX2
                clrf KEYSTORE
                call UPDATECLOCK
                call CLKSHOW
                return

CLKSHOW:        clrf ADRMSB
                movlw 0                 ; set column
                call LINE6              ; set cell number for line stated
                call SCREENADR          
                AWRON                   ; set auto write on
                call SENDCMD            ; send command

                movf CLKHRS,W           ; get hrs
                call LCDFRM             ; format and sent it to lcd
                movlw ':'               ; colon
                call LCDOUT             ; send to lcd
                movf CLKMIN,W           ; get mins
                call LCDFRM             ; format and send to lcd
                movlw '.'               ; decimal point
                call LCDOUT             ; send to lcd
                movf CLKSEC,W           ; get secs
                call LCDFRM             ; format and send to lcd

ENDSHW:         AWROFF                  ; auto write off
                call SENDCMD            ; send command
                return

; *********

UPDATECLOCK:    clrf BYTE4
                movlw SECAD0            
                movwf FSR               
                movf CLKAD0,W           ; secs add step lsb (decimal place)
                call DADD0              
                movlw SECAD1            
                movwf FSR               
                movf CLKAD1,W           ; secs add step nsb (decimal place)
                call DADD1              
                movlw SECAD2            
                movwf FSR               
                movf CLKAD2,W           ; secs add step msb (decimal place)
                call DADD2              
                movf BYTE4,W            ; is there an overflow i.e. has 1 sec been counted?
                btfsc STATUS,Z          
                return                  

SECCLK:         call TIDECOUNT
                call MOONCLOCK

                incf CLKSEC,F           ; inc secs in bcd
                movlw 6                 ; check units >9
                addwf CLKSEC,W          ; if 6 is added is there a digit carry?
                btfss STATUS,DC         
                goto ENDCLK
                movwf CLKSEC            ; yes
                xorlw %01100000         ; xor w with 96 (bcd 60)
                btfss STATUS,Z          ; is the answer = 0 (secs = 60)?
                goto ENDCLK
                clrf CLKSEC             ; yes, clear secs counter

MINCLK:         incf CLKMIN,F           ; inc mins in bcd
                movlw 6                 ; check units >9
                addwf CLKMIN,W          ; if 6 is added is there a digit carry?
                btfss STATUS,DC         
                goto ENDCLK
                movwf CLKMIN            ; yes
                xorlw 96                ; xor w with 96 (bcd 60)
                btfss STATUS,Z          ; is the answer = 0 (mins = 60)?
                goto ENDCLK
                clrf CLKMIN             ; yes, clear mins counter

HRSCLK:         incf CLKHRS,F           ; inc hrs in bcd
                movlw 6                 ; check units >9
                addwf CLKHRS,W          ; if 6 is added is there a digit carry?
                btfsc STATUS,DC         
                movwf CLKHRS            ; yes
                movf CLKHRS,W           
                xorlw %00100100         ; xor w with bcd 24 (check total=24)
                btfss STATUS,Z          ; is the answer = 1 (equality)
                goto ENDCLK
                clrf CLKHRS             ; yes, clear hours count

WEEKCLK:        incf WEEKDAY,F          ; inc weekday in binary
                movlw 7                 
                xorwf WEEKDAY,W         
                btfsc STATUS,Z          
                clrf WEEKDAY            

DAYCLK:         incf DAY,F              ; inc days in bcd
                movlw 6                 ; check units >9
                addwf DAY,W             ; if 6 is added is there a digit carry?
                btfsc STATUS,DC         
                movwf DAY               ; yes

                decf DAY,W
                xorwf MONTHLEN,W        ; xor w with month length
                btfss STATUS,Z          ; is the answer = 1 (equality)
                goto STORECLK
                movlw 1                 
                movwf DAY               ; yes, reset days count

MONTHCLK:       incf MONTH,F            ; inc month in binary
                movlw 12                ; December = 11
                xorwf MONTH,W           
                btfsc STATUS,Z          
                clrf MONTH              ; January = 0
                movlw 2
                movwf PCLATH
                bsf PCLATH,3
                call MONTH4             ; get days in month
                clrf PCLATH
                movwf MONTHLEN          
                call LEAPYEAR           ; correct for leap year?

YEARCLK:        movf MONTH,W            
                btfss STATUS,Z          
                goto STORECLK           
YEARCLK2:       incf YEAR,F             ; inc year in bcd
                movf YEAR,W
                movlw 6                 ; check units >9
                addwf YEAR,W            ; if 6 is added is there a digit carry?
                btfsc STATUS,DC         
                movwf YEAR              ; yes
                movf YEAR,W             ; is year = 100 ($A0)
                xorlw $A0
                btfss STATUS,Z
                goto STORECLK
                clrf YEAR

STORECLK:       call DATESHOW
                call STOREPROM

ENDCLK:         return

LEAPYEAR:       movf MONTH,W            ; is month = February ?
                xorlw 1                 
                btfss STATUS,Z          
                return                  ; no
                movlw $28               ; yes, set FEB for 28
                movwf MONTHLEN
                swapf YEAR,W            ; convert year bcd to decimal
                andlw %00001111
                movwf REGA0
                clrf REGA1
                clrf REGA2
                clrf REGA3
                movlw 10
                movwf REGB0
                clrf REGB1
                clrf REGB2
                clrf REGB3
                call MULTIPLY
                movf YEAR,W
                andlw %00001111
                addwf REGA0,W
                andlw %00000011
                btfsc STATUS,Z
                incf MONTHLEN,F        ; yes, set FEB for $29
                movf DAY,W             ; is current day count > monthlen?
                subwf MONTHLEN,W
                btfsc STATUS,C
                return                 ; no
                movf MONTHLEN,W        ; yes, so set day to monthlen
                movwf DAY
                call DATESHOW
                return

MOONCLOCK: decfsz MOONCOUNTLSB,F  ; 42524.05 mins per moon orbit (360 degrees)
           return                 ; x60 = 2551443.00 secs per orbit
           decfsz MOONCOUNTMSB,F  ; = 7087.3416 secs per moon degree
           return                 ; becomes 9966.5738 secs per bit of a 256 byte
           movf MOONRESETMSB,W    ; taken as 9967 ($26 EF) +1 on MSB ($27) because of decfsz
           movwf MOONCOUNTMSB     ; accuracy deviation = about 0.00428%
           movf MOONRESETLSB,W    ; Although, of course, Moon orbit timing
           movwf MOONCOUNTLSB     ; is not consistent in nature!
           incf MOONFACE,F        ; Nor is tide, of course
           call MOONSHOW       
           return

; *******************

TIDECOUNT:      decfsz TIDECOUNTLSB,F
                goto TC1
                decfsz TIDECOUNTMSB,F
                goto TC1

                movf TIDERESETMSB,W
                movwf TIDECOUNTMSB   ; (1 full tide cycle = 44714.166 secs = 12h 25m 14s)
                movf TIDERESETLSB,W
                movwf TIDECOUNTLSB   ; (fraction ignored) 44714 = $AEAA but because it is decfsz, byte vals become 1 greater, i.e.  $AFAB

                movlw $02            ; segments of tide for graph display
                movwf TIDESEGMENTMSB ; (44714/94)
                movlw $DC            ; 1 seg = 44714 / 94 = 475.681secs taken as 476secs = $01DC
                                     ; but because it is decfsz, MSB val becomes 1 greater, i.e.  $02DC
                movwf TIDESEGMENTLSB ; (94 = total segs used on LCD, 47 up, 47 down)
                goto TC1A

TC1:            decfsz TIDESEGMENTLSB,F
                goto TC2
                decfsz TIDESEGMENTMSB,F
                goto TC2
                movlw $02
                movwf TIDESEGMENTMSB
                movlw $DC
                movwf TIDESEGMENTLSB
                decfsz TIDEBASIC,F
                goto TC1A
                movlw 93
                movwf TIDEBASIC

TC1A:           movf TIDEBASIC,W
                addwf TIDEOFFSET,W
                movwf TIDEGRAPH
                addlw 163           ; is greater than 93?
                btfsc STATUS,C
                movwf TIDEGRAPH     ; yes, so put new val into TIDEGRAPH
                call SHOWTIDE
TC2:            return

; **************

CHECKTIDEUP:    clrf TIDEUP
                movf TIDEGRAPH,W
                movwf STORE
                addlw 209               ; is it greater than 47?
                btfss STATUS,C
                goto CHECKT2            ; no
                movlw 47                ; yes, subtract 47
                subwf STORE,F
                movf STORE,W            ; subtract from 47
                sublw 47
                movwf STORE
                bsf TIDEUP,7

CHECKT2:        movlw 1
                movwf REGA0
                clrf  REGA1
                clrf  REGA2
                clrf  REGA3
                movlw 128
                movwf REGB0
                clrf  REGB1
                clrf  REGB2
                clrf  REGB3
                return

WRITETIDELINE:  movlw 8
                movwf ADRLSB            ; set column
                movf LOOPE,W            ; set line
                call GLINE              ; multiply by line length to get address
                call SCREENADR
                AWRON                   ; set auto write on
                call SENDCMD            ; send command
                movf REGA3,W
                call AUTOWRITE 
                movf REGA2,W
                call AUTOWRITE 
                movf REGA1,W
                call AUTOWRITE 
                movf REGA0,W
                call AUTOWRITE 
                movf REGB0,W
                call AUTOWRITE 
                movf REGB1,W
                call AUTOWRITE 
                movf REGB2,W
                call AUTOWRITE 
                movf REGB3,W
                call AUTOWRITE 
                AWROFF                  ; set auto write off
                call SENDCMD            ; send command
                return

WRITETIDECLEAR: movlw 8
                movwf ADRLSB            ; set column
                movf LOOPE,W            ; set line
                call GLINE              ; multiply by line length to get address
                call SCREENADR          
                AWRON                   ; set auto write on
                call SENDCMD            ; send command
                movlw 8
                movwf LOOPD
STIDE2A:        movlw 0
                call AUTOWRITE 
                decfsz LOOPD,F
                goto STIDE2A
                AWROFF                  ; set auto write off
                call SENDCMD            ; send command
                return

ROTATETIDELINE: bsf STATUS,C
                rlf REGA0,F
                rlf REGA1,F
                rlf REGA2,F
                rlf REGA3,F
                bsf STATUS,C
                rrf REGB0,F
                rrf REGB1,F
                rrf REGB2,F
                rrf REGB3,F
                return

; *************

SHOWTIDE:       call CHECKTIDEUP
                clrf LOOPE
STIDE2:         movf LOOPE,W
                xorwf STORE,W
                btfsc STATUS,Z
                goto STIDE2B
                call WRITETIDECLEAR
                incf LOOPE,F
                movf LOOPE,W
                xorlw 47
                btfss STATUS,Z
                goto STIDE2
                goto STIDE4

STIDE2B:        btfsc TIDEUP,7          ; is tide rising?
                goto STIDE4             ; no

STIDE3:         call WRITETIDELINE
                call ROTATETIDELINE
                incf LOOPE,F
                movf LOOPE,W
                xorlw 47
                btfss STATUS,Z
                goto STIDE3
STIDE4:         movf TIDEGRAPH,W
                btfss STATUS,Z
                goto FTIDE54

                clrf ADRMSB
                movlw 9                 ; set column
                call LINE6              ; set text line
                call SCREENADR
                AWRON                   ; set auto write on
                call SENDCMD            ; send command
                movlw ' '
                call LCDOUT             ; send to lcd

                movlw 'B'
                btfss BIRDFLAG,0
                movlw ' '
                call LCDOUT

                movf CLKHRS,W           ; get hrs
                call LCDFRM             ; format and sent it to lcd
                movlw ':'               ; colon
                call LCDOUT             ; send to lcd
                movf CLKMIN,W           ; get mins
                call LCDFRM             ; format and send to lcd
                AWROFF                  ; auto write off
                call SENDCMD            ; send command
                btfsc BIRDFLAG,0        ; is bird wanted?
                goto BIRDSHOW           ; yes
                goto FTIDE54

; ************

SHOWTIDEFALL:   call CHECKTIDEUP
                btfsc TIDEUP,7          ; is tide rising?
                goto FTIDE5             ; no

                clrf LOOPE              ; yes
FTIDE2:         movf LOOPE,W
                xorwf STORE,W
                btfsc STATUS,Z
                goto FTIDE3
                call WRITETIDECLEAR
                incf LOOPE,F
                movf LOOPE,W
                xorlw 47
                btfss STATUS,Z
                goto FTIDE2
                goto FTIDE4

FTIDE3:         call WRITETIDELINE
                call ROTATETIDELINE
                incf LOOPE,F
                movf LOOPE,W
                xorlw 47
                btfss STATUS,Z
                goto FTIDE3
FTIDE4:         goto FTIDE54

FTIDE5:         clrf LOOPE
FTIDE52:        movf LOOPE,W
                xorwf STORE,W
                btfsc STATUS,Z
                goto FTIDE53
                call WRITETIDECLEAR
                call ROTATETIDELINE
                incf LOOPE,F
                movf LOOPE,W
                xorlw 47
                btfss STATUS,Z
                goto FTIDE52
                goto FTIDE54

FTIDE53:        call WRITETIDELINE
                call ROTATETIDELINE
                incf LOOPE,F
                movf LOOPE,W
                xorlw 47
                btfss STATUS,Z
                goto FTIDE53
FTIDE54:        clrf ADRMSB
                movlw 11                ; set column
                call LINE7              ; set text line
                call SCREENADR
                AWRON                   ; set auto write on
                call SENDCMD            ; send command
                movf TIDEGRAPH,W
                movwf REGA0
                clrf REGA1
                clrf REGA2
                clrf REGA3
                call BIN2DEC
                call SHOWDIG6
                AWROFF                  ; auto write off
                call SENDCMD            ; send command
                return

; **************

DADD0:          addwf INDF,F            ; adding routine for secs timing adjustment
                incf FSR,F              
                movf STATUS,W           
                andlw 1                 
DADD1:          addwf INDF,F            
                incf FSR,F              
                movf STATUS,W           
                andlw 1                 
DADD2:          addwf INDF,F            
                movf STATUS,W           
                andlw 1                 
                addwf BYTE4,F           
                return                  

; *************

LCDFRM:         movwf STORE2            ;split & format decimal byte for lcd
                swapf STORE2,W          ;swap byte into w to get tens
                andlw 15                ;and to get nibble
                iorlw 48                ;or with 48 to make ascii value
                call LCDOUT             ;send to lcd
                movf STORE2,W           ;get units
                andlw 15                ;and to get nibble
                iorlw 48                ;or with 48 to make ascii value
                call LCDOUT             ;send to lcd
                return                  

LCDOUT:         addlw 224               ; +224 is same as -32 for conversion from ascii
                call AUTOWRITE          ; auto write and increment
                return                  

; ************

DATESHOW:       clrf ADRMSB             
                movlw 0                 ; set column
                call LINE7              ; set cell number for line stated
                call SCREENADR          
                AWRON                   ; set auto write on
                call SENDCMD            ; send command

                movlw 2
                movwf PCLATH
                movf WEEKDAY,W          ; get weekday
                bsf PCLATH,3
                call DAY1
                clrf PCLATH
                call LCDOUT

                movlw 2
                movwf PCLATH
                movf WEEKDAY,W          ; get weekday
                bsf PCLATH,3
                call DAY2
                clrf PCLATH
                call LCDOUT

                movlw 2
                movwf PCLATH
                movf WEEKDAY,W          ; get weekday
                bsf PCLATH,3
                call DAY3
                clrf PCLATH
                call LCDOUT
                movlw ' '
                call LCDOUT             

                movf DAY,W              ; get day
                call LCDFRM             ; format and sent it to lcd

                movlw 2
                movwf PCLATH
                movf MONTH,W            ; get month
                bsf PCLATH,3
                call MONTH1
                clrf PCLATH
                call LCDOUT

                movlw 2
                movwf PCLATH
                movf MONTH,W            ; get month
                bsf PCLATH,3
                call MONTH2
                clrf PCLATH
                call LCDOUT

                movlw 2
                movwf PCLATH
                movf MONTH,W            ; get month
                bsf PCLATH,3
                call MONTH3
                clrf PCLATH
                call LCDOUT

                movf YEAR,W             ; get year
                call LCDFRM
                AWROFF                  ; auto write off
                call SENDCMD            ; send command
                return                  

                ;********** start of sub-routines ***************

PAUSIT:         movlw 5                 ; pause routine, 1/5th sec
                movwf CLKCNT            
                clrf INTCON             
PAUSE:          btfss INTCON,2          
                goto PAUSE              
                bcf INTCON,2            
                decfsz CLKCNT,F         
                goto PAUSE              
                return                  

                                        ;...............

                                        ;write data to eeprom routine modified for pic16f87x devices
                                        ;according to data sheet ds30292a page 43

                                        ;this routine is entered with w holding
                                        ;the eeprom byte address at which data
                                        ;is to be stored. the data to be stored
                                        ;is held in store1.
SETPRM:         bsf STATUS,RP1          ;set for page 2
                bcf STATUS,RP0          
                movwf EEADR             ;copy w into eeadr to set eeprom address
                bcf STATUS,RP1          ;set for page 0
                movf STORE1,W           ;get data value from store1 and hold in w
                bsf STATUS,RP1          ;set for page 2
                movwf EEDATA            ;copy w into eeprom data byte register
                bsf STATUS,RP0          ;set for page 3
                bcf EECON1,EEPGD        ;point to data memory
                bsf EECON1,WREN         ;enable write flag

MANUAL:         movlw $55               ;these lines cause the action required by
                movwf EECON2            ;by the eeprom to store the data in eedata
                movlw $AA               ;at the address held by eeadr.
                movwf EECON2            
                bsf EECON1,WR           ;set the ``perform write'' flag
                bcf STATUS,RP1          ;set for page 0
                bcf STATUS,RP0          

CHKWRT:         btfss PIR2,EEIF         ;wait until bit 4 of pir2 is set
                goto CHKWRT             
                bcf PIR2,EEIF           ;clear bit 4 of pir2
                return                  

                                        ;..............

INCPRM:         movwf STORE1            ;used for consecutive writes to eeprom
                movf LOOPB,W            
                incf LOOPB,F            
                goto SETPRM             

                                        ;..........

                                        ;read data from eeprom routine modified for pic16f87x devices
                                        ;according to data sheet ds30292a page 43

                                        ;this routine is entered with w holding
                                        ;the eeprom byte address to be read.
PRMGET:         bsf STATUS,RP1          ;set for page 2
                bcf STATUS,RP0          
                movwf EEADR             ;copy w into eeadr to set eeprom address
                bsf STATUS,RP0          ;set for page 3
                bcf EECON1,EEPGD        ;point to data memory
                bsf EECON1,RD           ;enable read flag
                bcf STATUS,RP0          ;set for page 2
                movf EEDATA,W           ;read eeprom data now in eedata into w
                bcf STATUS,RP1          ;set for page 0
                return                  


BITWRITE:                               ; ** write single single bit data routine **
                movwf BITVAL            
                call SCREENADR          ; set screen write address
                movf BITVAL,W           
                call SENDCMD            ; send command
                return                  ;

                                        ;.........

ONEWRITE:                               ;  ** write single byte **
                movwf ATTRIB            ; temp store val brought in on w
                call SCREENADR          ; set screen write address - vals preset at call
                AWRON                   ; auto write on
                call SENDCMD            ; send command
                call CHECK3             ; read status for da0/da1 = 3
                movf ATTRIB,W           
                call OUTDATA
                AWROFF                  ; auto write off
                call SENDCMD            ; send command
                return                  
                                        ;............

AUTOWRITE:                              ; ** auto write routine **
                movwf TEMPA             ; temp store value brought in on w
                call CHECK8             ; read status for da3 = 8
                movf TEMPA,W            ; write data
                call OUTDATA            ;
                return                  ;

                                        ;..............

CMDADR:                                 ; ** set address for command sending **
                call CHECK3             ; read status for da0/da1 = 3
                movf ADRLSB,W           ; write data d1
                call OUTDATA            
                call CHECK3             ; read status for da0/da1 = 3
                movf ADRMSB,W           ; write data d2
                call OUTDATA            
                return                  

                                        ;.........

SCREENADR:                              ; ** set address for write/read to/from screen
                call CHECK3             ; read status for da0/da1 = 3
                movf ADRLSB,W           ; write address lsb
                call OUTDATA            ;
                call CHECK3             ; read status for da0/da1 = 3
                movf ADRMSB,W           ; write address msb
                call OUTDATA            ;
                ADPSET                  ; set addrress pointer
                call SENDCMD            ; send command
                return                  ;

                                        ;.............

TEXTHOME:                               ; ** set text home address **
                clrf ADRMSB             ; text home address $0000
                clrf ADRLSB             ;
                call CMDADR             ; send command address
                TXHOME
                call SENDCMD            ; send command
                return                  
                                        ;...........

GRAPHHOME:                              ;  ** set graphic home address **
                movlw $02               ; graphic home address $0200
                movwf ADRMSB            
                clrf ADRLSB             
                call CMDADR             ; send command address
                GRHOME
                call SENDCMD            ; send command
                return                  

                                        ;...........

TEXTAREA:                               ; ** set text area **
                clrf ADRMSB             ;
                movf COLUMN,W           ; columns length
                movwf ADRLSB            ;
                call CMDADR             ; send command address
                TXAREA                  ; text area command
                call SENDCMD            ; send command
                return                  

                                        ;...........

GRAPHAREA:                              ; ** set graphic area **
                clrf ADRMSB             
                movf COLUMN,W           ; columns length
                movwf ADRLSB            
                call CMDADR             ; send command address
                GRAREA                  ; graphic area command
                call SENDCMD            ; send command
                return                  
                                        ;...........

SETMODE:                                ;  ** set mode - many options, see epe text **
                movlw %10000000         ; (or mode, internal cg mode)
                call SENDCMD            ; send command
                return                  
                                        ; %1000x000   ; or mode
                                        ; %1000x001   ; xor mode
                                        ; %1000x011   ; and mode
                                        ; %1000x100   ; text attribute mode
                                        ; %10000xxx   ; internal cg rom mode
                                        ; %10001xxx   ; external cg ram mode

                                        ;...........

SETOFFSET:                              ; ** set offset register **
                                        ; setting offset address at 2 selects
                                        ; cg ram start address = $1400 where
                                        ; $80 is character number of 1st graphic byte
                                        ; values below that call ascii text characters
                clrf ADRMSB             
                movlw 2                 
                movwf ADRLSB            
                call CMDADR             ; send command address
                OFFSET
                call SENDCMD            ; send command
                return                  

                                        ;...........

SETDISPLAY:                             ;  ** display mode **  some options:
                movlw %10010100         ; text on, graphic off, cursor & blink off
                call SENDCMD            ; send command
                return                  

                                        ;...........

CLRTXT:                                 ;  ** clear text area ($0000) **
                clrf ADRMSB             ; clear all text screen lines, length as set
                clrf ADRLSB             
                call SCREENADR          ; set screen write address
                AWRON                   ; auto write on
                call SENDCMD            ; send command
                movlw 8                 ; number of lines
                movwf LOOPC             
CLR2:           movf COLUMN,W           ; column length
                movwf LOOPB             ;
CLR3:           movlw 0                 ; write 0
                call AUTOWRITE          ; auto write and increment
                decfsz LOOPB,F          
                goto CLR3               
                decfsz LOOPC,F          
                goto CLR2               
                AWROFF                  ; auto write off
                call SENDCMD            ; send command
                return                  

                                        ;............

CLRGRAPH:                               ; ** clear graph area ($0200) **
                movlw $02               
                movwf ADRMSB            
                clrf ADRLSB             
                call SCREENADR          ; set screen write address
                AWRON                   ; auto write on
                call SENDCMD            ; send command
                movlw 64                ; number of lines
                movwf LOOPC             
CLRG2:          movf COLUMN,W           ; column length
                movwf LOOPB             
CLRG3:          movlw 0                 ; write 0
                call AUTOWRITE          ; auto write and increment
                decfsz LOOPB,F          
                goto CLRG3              
                decfsz LOOPC,F          
                goto CLRG2              
                AWROFF                  ; auto write off
                call SENDCMD            ; send command
                return                  

                                        ;............

SETATTR:                                ;  ** send attribute & other data to screen as 1 character, looped
                call SCREENADR          ; set screen write address - vals preset at call
                AWRON                   ; auto write on
                call SENDCMD            ; send command
SETAT:          movf ATTRIB,W           ; val to be sent preset at call
                call AUTOWRITE          ; auto write and increment
                decfsz LOOPC,F          ; loopc val specified by calling routine
                goto SETAT              
                AWROFF                  ; auto write off
                call SENDCMD            ; send command
                return                  

                                        ;............

CHECK3:         BANK1                   ; status check for da0/da1 = 3
                movlw 255               
                movwf TRISD             ; set portd as inputs
                BANK0                   ; rst  cd  ce  rd  wr
                movlw %00011001         ;  1   1   0   0   1
                movwf PORTC             ; set ce, rd low
                nop                     
CK3:            btfss PORTD,0           ; portd bit 0 set?
                goto CK3                ; no
CK3A:           btfss PORTD,1           ; portd bit 1 set?
                goto CK3A               ; no  rst  cd  ce  rd  wr
                movlw %00011111         ;      1   1   1   1   1
                movwf PORTC             ; set controls
                nop                     
                BANK1                   
                clrf TRISD              ; set portd as outputs
                BANK0                   
                return                  

                                        ;............

CHECK6:         BANK1                   ; status check for da6 low
                bsf TRISD,6             ; set portd bit as input
                BANK0                   ; rst  cd  ce  rd  wr
                movlw %00011001         ;  1   1   0   0   1
                movwf PORTC             ; set ce, rd low
                nop                     
CK6:            btfsc PORTD,6           ; is portd bit 6 low?
                goto CK6                ; no rst  cd  ce  rd  wr
                movlw %00011111         ;     1   1   1   1   1
                movwf PORTC             ; set controls
                nop                     
                BANK1                   
                bcf TRISD,6             ; set portd bit 6 as output
                BANK0                   
                return                  

                                        ;............

CHECK8:         BANK1                   ; status check for da3 = 8
                bsf PORTD,3             ; set portd bit 3 as input
                BANK0                   ; rst  cd  ce  rd  wr
                movlw %00011001         ;  1   1   0   0   1
                movwf PORTC             ; set ce, rd low
                nop                     
CK8:            btfss PORTD,3           ; is portd,3 high?
                goto CK8                ; no  rst  cd  ce  rd  wr
                movlw %00011111         ;      1   1   1   1   1
                movwf PORTC             ; set controls
                nop                     
                BANK1                   
                bcf TRISD,3             ; set portd bit 3 as output
                BANK0                   
                return                  

                                        ;........

OUTDATA:                                ; ** send data routine **
                movwf TEMPA             ; temp store val brought in on w
                                        ; rst  cd  ce  rd  wr
                movlw %00010111         ;  1   0   1   1   1
                movwf PORTC             ; set cd low
                movf TEMPA,W            ; get stored data
                movwf PORTD             ; send data
                nop                     ; rst  cd  ce  rd  wr
                movlw %00010010         ;  1   0   0   1   0
                movwf PORTC             ; set cd, ce, wr low
                nop                     ; rst  cd  ce  rd  wr
                movlw %00010111         ;  1   0   1   1   1
                movwf PORTC             ; set ce, wr high
                nop                     ; rst  cd  ce  rd  wr
                movlw %00011111         ;  1   1   1   1   1
                movwf PORTC             ; set cd high
                return                  

                                        ;..........

SENDCMD:                                ;  ** command write routine **
                movwf TEMPA             ; temp store val brought in on w
                call CHECK3             ; read status for da0/da1 = 3
                movf TEMPA,W            ; write command
                movwf PORTD             ; send stored data
                nop                     ; rst  cd  ce  rd  wr
                movlw %00011010         ;  1   1   0   1   0
                movwf PORTC             ; set ce, wr low
                nop                     ; rst  cd  ce  rd  wr
                movlw %00011111         ;  1   1   1   1   1
                movwf PORTC             ; set all high
                return                  

                                        ;............

LINE7:          addwf COLUMN,W          ; sets line addresses for text screen
                btfsc STATUS,C          
                incf ADRMSB,F           
LINE6:          addwf COLUMN,W          
                btfsc STATUS,C          
                incf ADRMSB,F           
LINE5:          addwf COLUMN,W          
                btfsc STATUS,C          
                incf ADRMSB,F           
LINE4:          addwf COLUMN,W          
                btfsc STATUS,C          
                incf ADRMSB,F           
LINE3:          addwf COLUMN,W          
                btfsc STATUS,C          
                incf ADRMSB,F           
LINE2:          addwf COLUMN,W          
                btfsc STATUS,C          
                incf ADRMSB,F           
LINE1:          addwf COLUMN,W          
                btfsc STATUS,C          
                incf ADRMSB,F           
LINE0:          movwf ADRLSB            ; column sets number of cells per line
                return                  ; adrmsb is set/cleared before routine called

                                        ;..............

GLINE:                                  ;** get graphic line address **
                movwf LOOPB             ; store line val
                movlw 2                 ; set graphic base address ($02xx)
                movwf ADRMSB            
                movf LOOPB,W            ; line val = 0?
                btfsc STATUS,Z          
                return                  ; yes
GLIN2:          movf COLUMN,W           ; no, multiply line length by line val
                addwf ADRLSB,F          
                movlw 1                 
                andwf STATUS,W          ; extract and add carry (if any) to msb
                addwf ADRMSB,F          
                decfsz LOOPB,F          
                goto GLIN2              
                return                  

                                        ;..........

SETUP:                                  ; rst  cd  ce  rd  wr  general setup
                movlw %00011111         ;  1   1   1   1   1
                movwf PORTC             ; set controls high (off)
                call TEXTHOME           ; set text home address
                call GRAPHHOME          ; set graphic home address
                call TEXTAREA           ; set text area
                call GRAPHAREA          ; set graphic area
                call SETMODE            ; set mode (int/ext/and-or-xor etc)
                call SETOFFSET          ; set offset (see epe text)
                call SETDISPLAY         ; display mode (text, graph on/off etc)
                call CLRTXT             ; write text blank code $0000
                call CLRGRAPH           ; write graph blank code $0200
                return                  

STOREPROM:      clrf LOOPB              ;store data to eeprom
                movf DAY,W              ;0
                call INCPRM             
                movf MONTH,W            ;1
                call INCPRM             
                movf YEAR,W             ;2
                call INCPRM             
                movf WEEKDAY,W          ;3
                call INCPRM             
                movf CLKAD0,W           ;4
                call INCPRM             
                movf CLKAD1,W           ;5
                call INCPRM             
                movf CLKAD2,W           ;6
                call INCPRM             
                movf CLKHRS,W           ;7 
                call INCPRM             
                movf CLKMIN,W           ;8 
                call INCPRM             
                movlw 0                 ;9  not used
                call INCPRM             
                movf TIDEOFFSET,W       ;10
                call INCPRM             
                movf MOONFACE,W         ;11
                call INCPRM             
                movf TIDEBASIC,W        ;12
                call INCPRM             
                movf TIDERESETMSB,W     ;13
                call INCPRM             
                movf TIDERESETLSB,W     ;14
                call INCPRM             
                movf MOONRESETMSB,W     ;15
                call INCPRM             
                movf MOONRESETLSB,W     ;16
                call INCPRM             

                return

; *********** PART OF PETER HEMSLEY'S 32-BIT MATHS ROUTINES *******

;*** SIGNED BINARY TO DECIMAL ***
;REGA -> DIGITS 1 (MSD) TO 10 (LSD) & DSIGN
;DSIGN = 0 if REGA is positive, FF if negative
;Return carry set if overflow
;Uses FSR register

bin2dec:
        call    clrdig          ;Clear all digits
        clrf    MTEMP           ;Reset sign flag
	call	chksgna		;Make REGA positive
	skpnc
        call    negatea         ;Negative

	movlw	D'32'		;Loop counter
	movwf	MCOUNT

b2dloop	rlf	REGA0,f		;Shift msb into carry
	rlf	REGA1,f
	rlf	REGA2,f
	rlf	REGA3,f

	movlw	DIGIT10
	movwf	FSR		;Pointer to digits
	movlw	D'10'		;10 digits to do
	movwf	DCOUNT

adjlp	rlf	INDF,f		;Shift digit and carry 1 bit left
        movlw   -D'10'
	addwf	INDF,w		;Check and adjust for decimal overflow
	skpnc
	movwf	INDF

	decf	FSR,f		;Next digit
	decfsz	DCOUNT,f
	goto	adjlp

	decfsz	MCOUNT,f	;Next bit
	goto	b2dloop

	btfsc	MTEMP,0		;Check sign
	comf	DSIGN,f		;Negative
	clrc

BLANKIT: movlw 48
        iorwf DIGIT1,F
        iorwf DIGIT2,F
        iorwf DIGIT3,F
        iorwf DIGIT4,F
        iorwf DIGIT5,F
        iorwf DIGIT6,F
        iorwf DIGIT7,F
        iorwf DIGIT8,F
        iorwf DIGIT9,F
        iorwf DIGIT10,F

        movlw 10          ; blank leading zeros
        movwf LOOP
        movlw DIGIT1
        movwf FSR
BLANK:  movf LOOP,W
        movf INDF,W
        andlw 15
        btfss STATUS,Z
        return
        bcf INDF,4
        incf FSR,F
        decfsz LOOP,F
        goto BLANK
        movlw 48
        iorwf DIGIT10,F
        return

; **************

;Check sign of REGA and convert negative to positive
;Used by multiply, divide, bin2dec

chksgna	rlf	REGA3,w
	skpc
	return

;Negate REGA
;Used by chksgna, multiply, divide, mod, bin2dec, dec2bin

NEGATEA: movf    REGA3,w         ;Save sign in w
	andlw	0x80

	comf	REGA0,f		;2's complement
	comf	REGA1,f
	comf	REGA2,f
	comf	REGA3,f
	incfsz	REGA0,f
	goto	nega1
	incfsz	REGA1,f
	goto	nega1
	incfsz	REGA2,f
	goto	nega1
	incf	REGA3,f
nega1
	incf	MTEMP,f		;flip sign flag
	addwf	REGA3,w		;Return carry set if -2147483648
	return


;Set all digits to 0
;Used by bin2dec

clrdig	clrf	DSIGN
	clrf	DIGIT1
	clrf	DIGIT2
	clrf	DIGIT3
	clrf	DIGIT4
	clrf	DIGIT5
	clrf	DIGIT6
	clrf	DIGIT7
	clrf	DIGIT8
	clrf	DIGIT9
	clrf	DIGIT10
	return

;Shift left REGA and REGC
;Used by multiply, divide

slac	rlf	REGA0,f
	rlf	REGA1,f
	rlf	REGA2,f
	rlf	REGA3,f
	rlf	REGC0,f
	rlf	REGC1,f
	rlf	REGC2,f
	rlf	REGC3,f
	return

;Check sign of REGB and negative convert to positive
;Used by multiply, divide, mod

chksgnb	rlf	REGB3,w
	skpc
	return			;Positive

;Negate REGB
;Used by chksgnb, subtract, multiply, divide, mod

negateb	movf	REGB3,w		;Save sign in w
	andlw	0x80

	comf	REGB0,f		;2's complement
	comf	REGB1,f
	comf	REGB2,f
	comf	REGB3,f
	incfsz	REGB0,f
	goto	negb1
	incfsz	REGB1,f
	goto	negb1
	incfsz	REGB2,f
	goto	negb1
	incf	REGB3,f
negb1
	incf	MTEMP,f		;flip sign flag
	addwf	REGB3,w		;Return carry set if -2147483648
	return

movac	movf	REGA0,w
	movwf	REGC0
	movf	REGA1,w
	movwf	REGC1
	movf	REGA2,w
	movwf	REGC2
	movf	REGA3,w
	movwf	REGC3
	return


;Add REGB to REGA (Unsigned)
;Used by add, multiply,

addba	movf	REGB0,w		;Add lo byte
	addwf	REGA0,f

	movf	REGB1,w		;Add mid-lo byte
	skpnc			;No carry_in, so just add
	incfsz	REGB1,w		;Add carry_in to REGB
	addwf	REGA1,f		;Add and propagate carry_out

	movf	REGB2,w		;Add mid-hi byte
	skpnc
	incfsz	REGB2,w
	addwf	REGA2,f

	movf	REGB3,w		;Add hi byte
	skpnc
	incfsz	REGB3,w
	addwf	REGA3,f
	return

;*** SIGNED SUBTRACT ***
;REGA - REGB -> REGA   ;Return carry set if overflow

subtract:
        call    negateb         ;Negate and add
	skpnc
	return			;Overflow

;*** SIGNED ADD ***
;REGA + REGB -> REGA
;Return carry set if overflow

add:    movf    REGA3,w         ;Compare signs
	xorwf	REGB3,w
	movwf	MTEMP

	call	addba		;Add REGB to REGA

	clrc			;Check signs
	movf	REGB3,w		;If signs are same
	xorwf	REGA3,w		;so must result sign
	btfss	MTEMP,7		;else overflow
	addlw	0x80
	return

;*** SIGNED DIVIDE ***
;REGA / REGB -> REGA
;Remainder in REGC
;Return carry set if overflow or division by zero

divide32: movf  REGB0,w         ;Trap division by zero
	iorwf	REGB1,w
	iorwf	REGB2,w
	iorwf	REGB3,w
	sublw	0
	skpnc
	return

	clrf	MTEMP
	call	chksgna		;Make dividend (REGA) positive
	skpnc
	return			;Overflow
	call	chksgnb		;Make divisor (REGB) positive
	skpnc
	return			;Overflow

	clrf	REGC0		;Clear remainder
	clrf	REGC1
	clrf	REGC2
	clrf	REGC3

	movlw	D'32'		;Loop counter
	movwf	MCOUNT

dvloop	call	slac		;Shift dividend (REGA) msb into remainder (REGC)

	movf	REGB3,w		;Test if remainder (REGC) >= divisor (REGB)
	subwf	REGC3,w
	skpz
	goto	dtstgt
	movf	REGB2,w
	subwf	REGC2,w
	skpz
	goto	dtstgt
	movf	REGB1,w
	subwf	REGC1,w
	skpz
	goto	dtstgt
	movf	REGB0,w
	subwf	REGC0,w
dtstgt	skpc			;Carry set if remainder >= divisor
	goto	dremlt

	movf	REGB0,w		;Subtract divisor (REGB) from remainder (REGC)
	subwf	REGC0,f
	movf	REGB1,w
	skpc
	incfsz	REGB1,w
	subwf	REGC1,f
	movf	REGB2,w
	skpc
	incfsz	REGB2,w
	subwf	REGC2,f
	movf	REGB3,w
	skpc
	incfsz	REGB3,w
	subwf	REGC3,f
	clrc
	bsf	REGA0,0		;Set quotient bit

dremlt	decfsz	MCOUNT,f	;Next
	goto	dvloop

	btfsc	MTEMP,0		;Check result sign
	call	negatea		;Negative
	return

;*** SIGNED MULTIPLY ***
;REGA * REGB -> REGA
;Return carry set if overflow

multiply
	clrf	MTEMP
	call	chksgna		;Make REGA positive
	skpnc
	return			;Overflow
	call	chksgnb		;Make REGB positive
	skpnc
	return			;Overflow

	call	movac		;Move REGA to REGC
	call	clra		;Clear product

	movlw	D'31'		;Loop counter
	movwf	MCOUNT

muloop	call	slac		;Shift left product and multiplicand

	rlf	REGA3,w
	skpnc			;Check for overflow
	return

	btfss	REGC3,7		;If multiplicand bit is a 1 then
	goto	nxtmul
	call	addba		;add multiplier to product

	rlf	REGA3,w
	skpnc			;Check for overflow
	return

nxtmul	decfsz	MCOUNT,f	;Next
	goto	muloop

	btfsc	MTEMP,0		;Check result sign
	call	negatea		;Negative
	return

;Clear REGA
;Used by multiply, sqrt

clra	clrf	REGA0
	clrf	REGA1
	clrf	REGA2
	clrf	REGA3
	return

        ;******* show moonface ******************

MOONCALL:       movlw %11111000     ; set bit (bits 0-2 hold the bit number)
                movwf MOONPHASE
                movf MOONFACE,W
                movwf MOONSTORE
                clrf MOONFACE

MOONLOOP:       call MOONSHOW
                incf MOONFACE,F
                movf MOONFACE,W
                xorwf MOONSTORE,W
                btfss STATUS,Z
                goto MOONLOOP
                return

MOONSHOW:       bsf MOONPHASE,3
                btfsc MOONFACE,7
                bcf MOONPHASE,3
                movf MOONFACE,W
                movwf B
                andlw %01000000
                movwf LEFTMOON
                btfss MOONFACE,6
                comf B,F
                movf B,W
                andlw 63
                movwf B

                movlw 1
                movwf ANGLE
MOONSIZE:       bsf PCLATH,3
                movf ANGLE,W            ; table counter
                call MCLOCK             ; get X value
                bcf PCLATH,3            
                movwf X       
                bcf STATUS,C 
                rrf X,W
                movwf X
                movwf REGA0
                clrf REGA1
                clrf REGA2
                clrf REGA3
                movf B,W
                movwf REGB0
                clrf REGB1
                clrf REGB2
                clrf REGB3
                call MULTIPLY
                movlw 64
                movwf REGB0
                clrf REGB1
                clrf REGB2
                clrf REGB3
                call DIVIDE32
                movf REGA0,W
                btfsc LEFTMOON,6
                goto MR1
                addlw 32
                goto MR2
MR1:            sublw 31
MR2:            movwf X
                bsf PCLATH,3
                bsf PCLATH,0
                movf ANGLE,W            ; table counter
                call MCLOCK2            ; get Y value
                clrf PCLATH
                movwf Y
                bcf STATUS,C 
                rrf Y,W
                movwf Y
                movlw 8
                subwf Y,F

                bcf STATUS,C     ; divide X by 8 to get byte for column number
                rrf X,W 
                movwf STORE
                bcf STATUS,C 
                rrf STORE,F
                bcf STATUS,C 
                rrf STORE,W
                movwf ADRLSB

                movf Y,W         ; from Y set line number
                call GLINE
                call SCREENADR          ; set screen write address
                comf X,W
                andlw %00000111
                iorwf MOONPHASE,W    ; set bit (bits 0-2 hold the bit number)
                call BITWRITE

                incf ANGLE,F
                movf ANGLE,W
                xorlw 90
                btfss STATUS,Z
                goto MOONSIZE
                return

; ************ KEYBOARD DETECTION ROUTINE **********

KEYBOARD:       call RECEIVE      ; get keyboard key pressed

                movf BYTE,W       ; is BYTE > 0
                btfsc STATUS,Z
                return

                btfss PAR,0       ; yes, is parity bit correct (hi)?
                return
                btfss PAR2,0      ; yes, so is 2nd parity bit correct (hi)?
                return

                movlw 2
                movwf PCLATH
                bsf PCLATH,3
                call TABLE1       ; get key path
                btfss STATUS,Z
                movwf KEYSTORE
                clrf PCLATH
                return

; ***********

SETLINE6:       clrf ADRMSB
                movlw 9                 ; set column
                call LINE6              ; set text line
                call SCREENADR
                AWRON                   ; set auto write on
                call SENDCMD            ; send command
                return

; ***********

SHOWCHECKMOON:  movlw 1
                movwf CHANGEVAL
                movlw 0
                goto SHOWCHECKNAME

SHOWCHECKMONTH: movlw 4
                movwf CHANGEVAL
                movlw 7
                goto SHOWCHECKNAME

SHOWCHECKDAY:   movlw 6
                movwf CHANGEVAL
                movlw 14
                goto SHOWCHECKNAME

SHOWCHECKTIDE:  movlw 2
                movwf CHANGEVAL
                movlw 21
                goto SHOWCHECKNAME

SHOWCHECKWEEKDAY: movlw 3
                movwf CHANGEVAL
                movlw 28
                goto SHOWCHECKNAME

SHOWCHECKHOUR:  movlw 7
                movwf CHANGEVAL
                movlw 35
                goto SHOWCHECKNAME

SHOWCHECKYEAR:  movlw 5
                movwf CHANGEVAL
                movlw 42
                goto SHOWCHECKNAME

SHOWCHECKMINUTES: movlw 8
                movwf CHANGEVAL
                movlw 49
                goto SHOWCHECKNAME

SHOWTIMEADJUST: movlw 9
                movwf CHANGEVAL
                movlw 56
                call SHOWCHECKNAME
                goto SHOWADJUST

SHOWCHECKSECS:  movlw 10
                movwf CHANGEVAL
                movlw 63
                goto SHOWCHECKNAME

SHOWTIDEADJUST: movlw 11
                movwf CHANGEVAL
                movlw 70
                call SHOWCHECKNAME
                goto SHOWTADJUST

SHOWMOONADJUST: movlw 12
                movwf CHANGEVAL
                movlw 77
                call SHOWCHECKNAME
                goto SHOWMADJUST

TOGGLEBIRD:     incf BIRDFLAG,F
                clrf ADRMSB
                movlw 10                ; set column
                call LINE6              ; set cell number for line stated
                call SCREENADR          
                AWRON                   ; set auto write on
                call SENDCMD            ; send command
                movlw 'B'
                btfss BIRDFLAG,0
                movlw ' '
                call LCDOUT
                AWROFF                  ; set auto write off
                call SENDCMD            ; send command
                return

SHOWEXIT:       call STOREPROM
                call CLRENDLINE6
                call CLRENDLINE7
                call SHOWTIDE
                clrf KEYSTORE
                clrf CHANGEVAL
                return                  ; end of show routine

SHOWCHECKNAME:  movwf NAMEVAL
                call SETLINE6
                movlw 8
                movwf LOOP
SHOWNAME2:      movf NAMEVAL,W
                call NAMETABLE
                call LCDOUT
                incf NAMEVAL,F
                decfsz LOOP,F
                goto SHOWNAME2

                AWROFF                  ; set auto write off
                call SENDCMD            ; send command

CLRENDLINE7:    call DATESHOW
                movlw 11                ; set column
                call LINE7              ; set cell number for line stated
                call SCREENADR          
                AWRON                   ; set auto write on
                call SENDCMD            ; send command
                movlw ' '
                call LCDOUT
                movlw ' '
                call LCDOUT
                movlw ' '
                call LCDOUT
                movlw ' '
                call LCDOUT
                movlw ' '
                call LCDOUT
                AWROFF                  ; set auto write off
                call SENDCMD            ; send command
                return   

CLRENDLINE6:    call CLKSHOW
                movlw 8                 ; set column
                call LINE6              ; set cell number for line stated
                call SCREENADR          
                AWRON                   ; set auto write on
                call SENDCMD            ; send command
                movlw ' '
                call LCDOUT
                movlw ' '
                call LCDOUT
                movlw ' '
                call LCDOUT
                movlw ' '
                call LCDOUT
                movlw ' '
                call LCDOUT
                movlw ' '
                call LCDOUT
                movlw ' '
                call LCDOUT
                movlw ' '
                call LCDOUT
                AWROFF                  ; set auto write off
                call SENDCMD            ; send command
                return   

; ****************

INCMOON:        incf MOONFACE,F
                call MOONSHOW
                return   

DECMOON:        movlw %10000000
                addwf MOONFACE,F
                call MOONSHOW
                decf MOONFACE,F
                movlw %10000000
                addwf MOONFACE,F
                return   

INCTIDE:        decfsz TIDEOFFSET,F
                goto INCT2
                movlw 93
                movwf TIDEOFFSET

INCT2:          movf TIDEBASIC,W
                addwf TIDEOFFSET,W
                movwf TIDEGRAPH
                addlw 163           ; is greater than 93?
                btfsc STATUS,C
                movwf TIDEGRAPH     ; yes, so put new val into TIDEGRAPH
                call SHOWTIDE
                return   

DECTIDE:        incf TIDEOFFSET,F
                movf TIDEOFFSET,W
                xorlw 95
                btfsc STATUS,Z
                clrf TIDEOFFSET

                movf TIDEBASIC,W
                addwf TIDEOFFSET,W
                movwf TIDEGRAPH
                addlw 163           ; is greater than 93?
                btfsc STATUS,C
                movwf TIDEGRAPH     ; yes, so put new val into TIDEGRAPH
                call SHOWTIDEFALL
                return   

INCMONTH:   incf MONTH,F
            movf MONTH,W
            xorlw 12
            btfsc STATUS,Z
            clrf MONTH  

            movlw 2
            movwf PCLATH
            bsf PCLATH,3
            call MONTH4             ; get days in month
            clrf PCLATH
            movwf MONTHLEN
            call LEAPYEAR
            call DATESHOW
            return   

DECMONTH:   movf MONTH,W
            btfss STATUS,Z
            goto DECM1
            movlw 11
            movwf MONTH
            goto DECM2

DECM1:      decf MONTH,F
           
DECM2:      movlw 2
            movwf PCLATH
            bsf PCLATH,3
            call MONTH4             ; get days in month
            clrf PCLATH
            movwf MONTHLEN
            call LEAPYEAR

DECMON2:    call DATESHOW
            return   

INCCALENDAR: call REGISTERTABLE
            movwf FSR
            call LIMITTABLE
            xorwf INDF,W
            btfss STATUS,Z        ; is the answer = 0
            goto INCCAL1
            clrf INDF
            goto INCCAL2

INCCAL1:    incf INDF,F
            movlw 6               ; check units >9
            addwf INDF,W          ; if 6 is added is there a digit carry?
            btfsc STATUS,DC         
            movwf INDF            ; yes

INCCAL2:    movf DAY,W            ; avoid zero for days
            btfsc STATUS,Z
            incf DAY,F
            call LEAPYEAR
            call DATESHOW
            return   

DECCALENDAR: call REGISTERTABLE
            movwf FSR
            movf INDF,W
            btfss STATUS,Z
            goto DECCAL1
            call LIMITTABLE
            movwf INDF
            goto DECCAL2

DECCAL1:    decf INDF,F
            movlw 6               ; check units >9
            addwf INDF,W          ; if 6 is added is there a digit carry?
            btfss STATUS,DC
            goto DECCAL2 
            movf INDF,W
            andlw %11110000
            iorlw 9
            movwf INDF            ; yes

DECCAL2:    movf DAY,W            ; avoid zero for days
            btfss STATUS,Z
            goto DECCAL3
            movf MONTHLEN,W
            movwf DAY
DECCAL3:    call LEAPYEAR
            call DATESHOW
            return   

INCCLOCK:   call REGISTERTABLE
            movwf FSR
            call LIMITTABLE
            xorwf INDF,W
            btfss STATUS,Z        ; is the answer = 0
            goto INCCLK1
            clrf INDF
            goto INCCLK2

INCCLK1:    incf INDF,F
            movlw 6               ; check units >9
            addwf INDF,W          ; if 6 is added is there a digit carry?
            btfsc STATUS,DC         
            movwf INDF            ; yes

INCCLK2:    call CLKSHOW
            return   

DECCLOCK:   call REGISTERTABLE
            movwf FSR
            movf INDF,W
            btfss STATUS,Z
            goto DECCLK1
            call LIMITTABLE
            movwf INDF
            goto DECCLK2

DECCLK1:    decf INDF,F
            movlw 6               ; check units >9
            addwf INDF,W          ; if 6 is added is there a digit carry?
            btfss STATUS,DC
            goto DECCLK2 
            movf INDF,W
            andlw %11110000
            iorlw 9
            movwf INDF            ; yes

DECCLK2:    call CLKSHOW
            return   

ZEROSECS:   clrf CLKSEC
            call CLKSHOW
            return   

; ********* adjust clock's timekeeping *********

INCADJUST:      incfsz CLKAD0,F
                goto SHOWADJUST
                incfsz CLKAD1,F
                goto SHOWADJUST
                incf CLKAD2,F
                goto SHOWADJUST

DECADJUST:      decf CLKAD0,F
                incfsz CLKAD0,W
                goto SHOWADJUST
                decf CLKAD1,F
                incfsz CLKAD1,W
                goto SHOWADJUST
                decf CLKAD2,F

SHOWADJUST:     movf CLKAD0,W
                movwf REGA0
                movf CLKAD1,W
                movwf REGA1
                movf CLKAD2,W
                movwf REGA2

                btfsc CLKAD2,7
                goto SHWADJ2
                call NEGATEA

SHWADJ2:        clrf REGA3
                bcf REGA2,7
                call BIN2DEC
                clrf ADRMSB
                movlw 9                 ; set column
                call LINE7              ; set cell number for line stated
                call SCREENADR          
                AWRON                   ; set auto write on
                call SENDCMD            ; send command
                call SHOWDIG5
                movlw '-'
                btfsc CLKAD2,7
                movlw '+'               
                call LCDOUT             
                AWROFF                  ; auto write off
                call SENDCMD            ; send command
                return   


; ********* adjust tide's timekeeping *********

INCTADJUST:     incfsz TIDERESETLSB,F
                goto SHOWTADJUST
                incf TIDERESETMSB,F
                goto SHOWTADJUST

DECTADJUST:     decf TIDERESETLSB,F
                incfsz TIDERESETLSB,W
                goto SHOWTADJUST
                decf TIDERESETMSB,F

SHOWTADJUST:    movf TIDERESETLSB,W
                movwf REGA0
                decf TIDERESETMSB,W
                movwf REGA1

SHWTADJ2:       clrf REGA2
                clrf REGA3
                bcf REGA2,7
                call BIN2DEC
                clrf ADRMSB
                movlw 9                 ; set column
                call LINE7              ; set cell number for line stated
                call SCREENADR          
                AWRON                   ; set auto write on
                call SENDCMD            ; send command
                call SHOWDIG4
                AWROFF                  ; auto write off
                call SENDCMD            ; send command
                return

; ********* adjust moon's timekeeping *********

INCMADJUST:     incfsz MOONRESETLSB,F
                goto SHOWMADJUST
                incf MOONRESETMSB,F
                goto SHOWMADJUST

DECMADJUST:     decf MOONRESETLSB,F
                incfsz TIDERESETLSB,W
                goto SHOWMADJUST
                decf MOONRESETMSB,F

SHOWMADJUST:    movf MOONRESETLSB,W
                movwf REGA0
                decf MOONRESETMSB,W
                movwf REGA1
                goto SHWTADJ2

; ********* RECEIVE KEYBOARD PRESS ******

RECEIVE:        clrf BYTE
                clrf BYTE2
                movlw 8           ; number of bits
                movwf LOOP
                clrf PAR          ; clear parity record for BYTE
                movlw 1           ; but set parity record for BYTE2
                movwf PAR2

RX1:            btfss PORTA,1     ; wait till clock high (idle clock)
                goto RX1
                btfsc PORTA,0     ; is data line low (start bit low)?
                return            ; no it's high, so a false Start bit, restart

RECDATA:        rrf BYTE,F        ; rotate byte right
                call HIGHLOW      ; wait for clk hi to lo transition
RX2:            btfsc PORTA,0     ; is data low? 
                goto RECSET       ; no, it's high
                bcf BYTE,7        ; yes, data is low so clear bit 7 of BYTE
                goto RECNEXT      ; and ignore parity counter
      
RECSET:         bsf BYTE,7        ; set bit 7 of BYTE
                incf PAR,F        ; inc parity counter
RECNEXT:        decfsz LOOP,F
                goto RECDATA      ; loop until 8 bits have been recd
                call HIGHLOW      ; wait for clk hi to lo transistion

                movf PORTA,W      ; get parity bit
                xorwf PAR,W       ; XOR with parity counter
                andlw 1           ; extract bit 0 of parity
                movwf PAR         ; and store it

                call HIGHLOW      ; get stop bit

                movf BYTE,W       ; is first byte $E0 (1st part of 2-byte set)?
                xorlw $E0
                btfsc STATUS,Z
                call RECEIVE2     ; yes, so get second byte

                movf BYTE,W       ; is first byte $E1 (1st part of 2-byte set)?
                xorlw $E1
                btfss STATUS,Z
                goto RX2A
                call RECEIVE2     ; yes, so get second byte
                goto RX4

RX2A:           movf BYTE,W       ; is first byte $F0 (1st part of 2-byte set)?
                xorlw $F0
                btfsc STATUS,Z
                call RECEIVE2     ; yes, so get second byte

RX4:            return

; ************** get high to low transition ********

HIGHLOW:        btfss PORTA,1     ; loop until CLK high
                goto HIGHLOW
HL2:            btfsc PORTA,1     ; loop until CLK low
                goto HL2
                return

; ********* RECEIVE 2nd KEYBOARD BYTE (if 1st byte = E0, F0, E1) ******

RECEIVE2:       movf BYTE,W       ; copy BYTE & PAR into BYTE2 & PAR2
                movwf BYTE2
                movf PAR,W
                movwf PAR2
                clrf PAR          ; clear parity record

                call HIGHLOW      ; get start bit

                clrf BYTE
                movlw 8           ; number of bits
                movwf LOOP

RECDATA2:       rrf BYTE,F        ; rotate byte right
                call HIGHLOW      ; wait for hi to lo transistion
RX22:           btfsc PORTA,0     ; is data low?
                goto RECSET2      ; no, it's high
                bcf BYTE,7        ; yes; it's high so clear bit 7 of BYTE
                goto RECNEXT2     ; and ignore parity counter
      
RECSET2:        bsf BYTE,7        ; set bit 7 of byte
                incf PAR,F        ; inc parity counter
RECNEXT2:       decfsz LOOP,F
                goto RECDATA2     ; loop until 8 bits have been recd

                call HIGHLOW      ; wait for clk hi to lo transistion
                movf PORTA,W      ; get parity bit
                xorwf PAR,W       ; XOR with parity counter
                andlw 1           ; extract bit 0 of parity
                movwf PAR         ; and store it
                return            ; end of 2nd byte key get routine

; **********

SHOWDIG1: movf DIGIT1,W
          call LCDOUT
SHOWDIG2: movf DIGIT2,W
          call LCDOUT
SHOWDIG3: movf DIGIT3,W
          call LCDOUT
SHOWDIG4: movf DIGIT4,W
          call LCDOUT
SHOWDIG5: movf DIGIT5,W
          call LCDOUT
SHOWDIG6: movf DIGIT6,W
          call LCDOUT
SHOWDIG7: movf DIGIT7,W
          call LCDOUT
SHOWDIG8: movf DIGIT8,W
          call LCDOUT
SHOWDIG9: movf DIGIT9,W
          call LCDOUT
SHOWDIG10: movf DIGIT10,W
          call LCDOUT
          return

    ;******* BIRD SHOW ***

BIRDSHOW: CLRF LOOPD
          CLRF LOOPE          
          CLRF LOOPF          

BIRDX:    MOVF LOOPE,W        
          addlw 8
          MOVWF ADRLSB        ; set column
          MOVLW 15            ; set line
          CALL GLINE          ; multiply by line length to get address

BIRD2:    MOVLW 32            ; get & show BIRD details
          MOVWF LOOPB         
BIRD2A:   movlw 3
          movwf PCLATH
          bsf PCLATH,3
          movf LOOPF,W
          call DUCK
          movwf STORE
          comf STORE,W
          clrf PCLATH
          incf LOOPF,F
          CALL ONEWRITE
          MOVF COLUMN,W       
          ADDWF ADRLSB,F      
          BTFSC STATUS,C      
          INCF ADRMSB,F       
          DECFSZ LOOPB,F      
          GOTO BIRD2A         
          INCF LOOPE,F        
          movf LOOPE,W
          xorlw 7
          btfss STATUS,Z
          goto BIRDX          
          goto FTIDE54


    ; ***** moon tables imported to here ******

                .ORG $0800

MCLOCK:         addwf PCL,F             
                include "MoonClockX.inc"    ; (X value)

                ORG $0900

MCLOCK2:        addwf PCL,F             
                include "MoonClockY.inc"    ; (Y value)

                org $0A00

DAY1:           addwf PCL,F             
                retlw 'M'
                retlw 'T'               
                retlw 'W'               
                retlw 'T'               
                retlw 'F'               
                retlw 'S'               
                retlw 'S'               
                retlw ''               

DAY2:           addwf PCL,F             
                retlw 'O'
                retlw 'U'               
                retlw 'E'               
                retlw 'H'               
                retlw 'R'               
                retlw 'A'               
                retlw 'U'               
                retlw '#'               

DAY3:           addwf PCL,F
                retlw 'N'
                retlw 'E'               
                retlw 'D'               
                retlw 'U'               
                retlw 'I'               
                retlw 'T'               
                retlw 'N'               
                retlw '@'               

MONTH1:         addwf PCL,F             
                retlw 'J'               
                retlw 'F'               
                retlw 'M'               
                retlw 'A'               
                retlw 'M'               
                retlw 'J'               
                retlw 'J'               
                retlw 'A'               
                retlw 'S'               
                retlw 'O'               
                retlw 'N'
                retlw 'D'               
                retlw '*'               
                retlw '*'               
                retlw '*'
                retlw '*'               

MONTH2:         addwf PCL,F             
                retlw 'A'               
                retlw 'E'               
                retlw 'A'               
                retlw 'P'               
                retlw 'A'               
                retlw 'U'               
                retlw 'U'               
                retlw 'U'               
                retlw 'E'               
                retlw 'C'               
                retlw 'O'
                retlw 'E'               
                retlw '*'               
                retlw '*'               
                retlw '*'
                retlw '*'               

MONTH3:         addwf PCL,F             
                retlw 'N'               
                retlw 'B'               
                retlw 'R'               
                retlw 'R'               
                retlw 'Y'               
                retlw 'N'               
                retlw 'L'               
                retlw 'G'               
                retlw 'P'               
                retlw 'T'               
                retlw 'V'
                retlw 'C'               
                retlw '*'               
                retlw '*'               
                retlw '*'
                retlw '*'               

MONTH4:         movf MONTH,W            ; vals in bcd
                andlw 15                
                addwf PCL,F             
                retlw $31       ; 0  JAN
                retlw $28       ; 1  FEB
                retlw $31       ; 2  MAR
                retlw $30       ; 3  APR
                retlw $31       ; 4  MAY
                retlw $30       ; 5  JUN
                retlw $31       ; 6  JUL
                retlw $31       ; 7  AUG
                retlw $30       ; 8  SEP
                retlw $31       ; 9  OCT
                retlw $30       ; 10 NOV
                retlw $31       ; 11 DEC
                retlw '*'               
                retlw '*'               
                retlw '*'
                retlw '*'               

TABLE1:     movlw 160     ; ignore values above $5F
            addwf BYTE,W
            btfsc STATUS,C
            return   ;retlw 0
            movlw 21     ; ignore 1st 21 keycode vals
            subwf BYTE,W
            addwf PCL,F  ; from Keyboard demo prog
            retlw 0      ; 15  q 
            retlw 0      ; 16  1
            retlw 0      ; 17
            retlw 0      ; 18
            retlw 0      ; 19
            retlw 14     ; ZEROSECS ; 1A  z
            retlw 10     ; SHOWCHECKSECS ; 1B  s
            retlw 0      ; 1C  a
            retlw 3      ; SHOWCHECKWEEKDAY  ; 1D w
            retlw 0      ; 1E  2
            retlw 0      ; 1F
                  
            retlw 0      ; 20
            retlw 16     ; SHOWTIDEADJUST ;21  c
            retlw 0      ; 22  x
            retlw 6      ; SHOWCHECKDAY  ; 23  d
            retlw 0      ; 24  e
            retlw 0      ; 25  4
            retlw 0      ; 26  3
            retlw 0      ; 27  RWINDOW
            retlw 0      ; 28
            retlw 0      ; 29  space bar
            retlw 0      ; 2A  v
            retlw 0      ; 2B  f
            retlw 2      ; SHOWCHECKTIDE ; 2C  t
            retlw 0      ; 2D  r
            retlw 9      ; SHOWTIMEADJUST ; 2E  5 (%)
            retlw 0      ; 2F

            retlw 0      ; 30
            retlw 0      ; 31 n
            retlw 18     ; TOGGLEBIRD 32 b
            retlw 7      ; SHOWCHECKHOUR  ; 33 h
            retlw 0      ; 34 g
            retlw 5      ; SHOWCHECKYEAR  ; 35 y
            retlw 0      ; 36 6
            retlw 0      ; 37
            retlw 0      ; 38
            retlw 0      ; 39
            retlw 8      ; SHOWCHECKMINUTES ; 3A m
            retlw 0      ; 3B j
            retlw 0      ; 3C u
            retlw 0      ; 3D 7
            retlw 0      ; 3E 8
            retlw 0      ; 3F
                  
            retlw 0      ; 40
            retlw 0      ; 41 , comma
            retlw 0      ; 42 k
            retlw 0      ; 43 i
            retlw 17     ; SHOWTIDEADJUST ; 44 o (O for Orbit)
            retlw 0      ; 45 0
            retlw 0      ; 46 9
            retlw 0      ; 47
            retlw 0      ; 48
            retlw 0      ; 49 . full stop
            retlw 13     ; SHOWINDEX ; 4A  / (?)
            retlw 4      ; SHOWCHECKMONTH ; 4B l (L for Lunar)
            retlw 0      ; 4C  semicolon (;)
            retlw 0      ; 4D p
            retlw 12     ; MINUSKEYTABLE ; 4E  - minus
            retlw 0      ; 4F

            retlw 0      ; 50
            retlw 0      ; 51
            retlw 0      ; 52  ' apostrophe
            retlw 0      ; 53
            retlw 0      ; 54  [
            retlw 11     ; PLUSKEYTABLE  ; 55 = (+)
            retlw 0      ; 56
            retlw 0      ; 57
            retlw 0      ; 58  caps lock
            retlw 0      ; 59  right shift
            retlw 15     ; SHOWEXIT ; 5A  ENTER
            retlw 0      ; 5B  ]
            retlw 0      ; 5C
            retlw 1      ; SHOWCHECKMOON ; 5D # hash
            retlw 0      ; 5E
            retlw 0      ; 5F

; **************************

          ORG $0B00
DUCK:     addwf PCL,F
          include "moonclockduck4.inc"

; **************************

          ORG $2100

        DE   $01   ; 0  day   in bcd
        DE   $02   ; 1  month in bcd
        DE   $04   ; 2  year  in bcd
        DE   $00   ; 3  weekday

        DE   $00   ; 4  CLKAD0
        DE   $00   ; 5  CLKAD1
        DE   $80   ; 6  CLKAD2
        DE   $00   ; 7  CLKHRS in bcd

        DE   $00   ; 8  CLKMIN in bcd
        DE   0     ; 9  not used    
        DE   10    ; 10 TIDEOFFSET
        DE    1    ; 11 MOONFACE

        DE   13    ; 12 TIDEBASIC
        DE   $AF   ; 13 TIDERESETMSB
        DE   $AA   ; 14 TIDERESETLSB      

        DE   $27   ; 15 MOONRESETMSB
        DE   $EF   ; 16 MOONRESETLSB
        DE   0     ;
        DE   0     ;
        DE   0     ;

                .END


