////////////////////////////////////////////////////////////////////////////
////                          LCD_2x16.C                                ////
////                 Driver for common LCD modules                      ////
////                                                                    ////
////  lcd_init()   Must be called before any other function.            ////
////                                                                    ////
////  lcd_putc(c)  Will display c on the next position of the LCD.      ////
////                     The following have special meaning:            ////
////                      \f  Clear display                             ////
////                      \n  Go to start of second line                ////
////                      \b  Move back one position                    ////
////                      \1..2 clear line                              ////
////                                                                    ////
////  lcd_gotoxy(x,y) Set write position on LCD (upper left is 1,1)     ////
////                                                                    ////
////  lcd_getc(x,y)   Returns character at position x,y on LCD          ////
////                                                                    ////
////////////////////////////////////////////////////////////////////////////
////        (C) Copyright 1996,1997 Custom Computer Services            ////
//// This source code may only be used by licensed users of the CCS C   ////
//// compiler.  This source code may only be distributed to other       ////
//// licensed users of the CCS C compiler.  No other use, reproduction  ////
//// or distribution is permitted without written permission.           ////
//// Derivative programs created using this software in object code     ////
//// form are not restricted in any way.                                ////
////////////////////////////////////////////////////////////////////////////

// As defined in the following structure the pin connection is as follows:
//
//     C7  enable
//     C5  rs
//     C6  rw
//
//     B4  D4
//     B5  D5
//     B6  D6
//     B7  D7
//
//   LCD pins D0-D3 are not used and PIC B3 is not used.

//   mapped such to allow a keyboard on port B

struct lcd_pin_map_c 
        {                            //      This structure is overlayed
           boolean unused1;          // C0    
           boolean unused2;          // C1    
           boolean unused3;          // C2    
           boolean unused4;          // C3    
           boolean unused5;          // C4    
           boolean rs;               // C5   
           boolean rw;               // C6   
           boolean enable;           // C7   
        } lcd_c;

struct lcd_pin_map_b 
        {                            
           boolean unused3;          // B0   
           boolean unused4;          // B1   
           boolean unused5;          // B2   
           boolean unused6;          // B3   
           int     data : 4;         // B4 to B7    Data Bits.
        } lcd_b;

#byte lcd_c = 7                      // This puts the entire structure
                                     // on to port C (at address 7)
#byte lcd_b = 6                      // This puts the entire structure
                                     // on to port B (at address 6)

#define lcd_type 2           // 0=5x7, 1=5x10, 2=2 lines, 4=4 lines
#define lcd_line_two   0x40  // LCD RAM address for the second line

byte CONST LCD_INIT_STRING[4] = {0x20 | (lcd_type << 2), 0xc, 1, 6};
                             // These bytes need to be sent to the LCD
                             // to start it up.


                             // The following are used for setting
                             // the I/O port direction register.

STRUCT lcd_pin_map_b const LCD_WRITE = {1,1,1,1,0}; // For write mode all pins are out
STRUCT lcd_pin_map_b const LCD_READ = {1,1,1,1,15}; // For read mode data pins are in


byte lcd_read_byte() 
  {
      byte low,high;

      set_tris_c(0x08);
      set_tris_b(LCD_READ);
      lcd_c.rw = 1;
      delay_cycles(1);
      lcd_c.enable = 1;
      delay_cycles(1);
      high = lcd_b.data;
      lcd_c.enable = 0;
      delay_cycles(1);
      lcd_c.enable = 1;
      delay_us(1);
      low = lcd_b.data;
      lcd_c.enable = 0;
      set_tris_b(LCD_WRITE);
      return( (high<<4) | low);
  }


void lcd_send_nibble( byte n ) 
  {
      lcd_b.data = n;
      delay_cycles(1);
      lcd_c.enable = 1;
      delay_us(2);
      lcd_c.enable = 0;
  }


void lcd_send_byte( byte address, byte n ) 
  {
      lcd_c.rs = 0;
      while ( bit_test(lcd_read_byte(),7) ) ;
      lcd_c.rs = address;
      delay_cycles(1);
      lcd_c.rw = 0;
      delay_cycles(1); 
      lcd_c.enable = 0;
      lcd_send_nibble(n >> 4);
      lcd_send_nibble(n & 0xf);
  }


void lcd_init() 
  {
    byte i;

    set_tris_c(0x08);
    set_tris_b(LCD_WRITE);
    lcd_c.rs = 0;
    lcd_c.rw = 0;
    lcd_c.enable = 0;
    delay_ms(15);
    for(i=1;i<=3;++i) 
      {
       lcd_send_nibble(3);
       delay_ms(5);
      }
    lcd_send_nibble(2);
    for(i=0;i<=3;++i)
       lcd_send_byte(0,LCD_INIT_STRING[i]);
  }


void lcd_gotoxy( byte x, byte y) 
  {
   byte address;

   if(y<2)  {address=0;}
   if(y==2) {address=lcd_line_two;}
   if(y>2)  {address=0;}
   address+=x-1;
   lcd_send_byte(0,0x80|address);
  }

void lcd_putc( char c) 
  {int tmp;
   switch (c) 
     {
     case '\f'   : lcd_send_byte(0,1);
                   delay_ms(2);
                   break;
     case '\n'   : lcd_gotoxy(1,2);        break;
     case '\1'   : for (tmp=1;tmp<21;tmp++)
                      {lcd_gotoxy(tmp,1);
                       lcd_send_byte(1,0x20);
                      }
                   lcd_gotoxy(1,1);
                   break;
     case '\2'   : for (tmp=1;tmp<21;tmp++)
                      {lcd_gotoxy(tmp,2);
                       lcd_send_byte(1,0x20);
                      }
                   lcd_gotoxy(1,2);
                   break;
     case '\b'   : lcd_send_byte(0,0x10);  break;
     default     : lcd_send_byte(1,c);     break;
   }
}

void lcd_put_bcd( int c) 
  {
   int tmp;
    tmp = ((c & 0xF0)/0x10)+0x30;
    c = (c & 0x0F)+0x30;
    lcd_send_byte(1,tmp);
    lcd_send_byte(1,c);
   return;
  }

char lcd_getc( byte x, byte y) 
  {
   char value;

   lcd_gotoxy(x,y);
   lcd_c.rs=1;
   value = lcd_read_byte();
   lcd_c.rs=0;
   return(value);
  }

