; Program 4
; Demo of RBIF interrupt
; Turn on led if RB7 or RB6 are zero

	list	p=16F84,r=dec
	__config	h'3FF3'

; Macros

#define BANK0   BCF 0x03,5
#define BANK1   BSF 0x03,5

; Equates for registers

	include	p16f84.inc

; Data locations

SAVEW:	equ	0x20		; preserve W during interrupts
SAVES:	equ	0x21		; preserve STATUS during interrupts
COUNT:	equ	0x22		; count of timer ticks
ICOUNT	equ	0x23		; number of interrupts
EBYTE:	equ	0x24		; byte read from EEPROM
SOFTB:	equ	0x26		; soft copy of PORTB register
BPORT:	equ	0x27		; working copy of PORTB

; code
	org	0		; reset vector
	goto	init
	org	4
	goto	ISR		; Interrupt vector
	org	5

; initialise PIC

init:	clrf PORTA		; initialise all port outputs to zero
	clrf PORTB
	BANK1
	clrf	TRISA		; RA0 - RA4 all outputs
	movlw	b'11110001'	; RB7-4 inputs, RB3-1 outputs, RB0 input
	movwf	TRISB
	movlw	b'00000111'	; enable PORTB pullups, PSA to TMR, prescale /256
	movwf	OPTION_REG
	BANK0

; interrupt setup

	movf	PORTB,W		; clear initial mismatch on PORTB
	movwf	SOFTB		; and initialise soft copy
	bcf	INTCON,RBIF	; and ensure flag bit is clear
	bsf	INTCON,RBIE	; enable RBIF interrupt
	bsf	INTCON,GIE	; enable global interrupts

; data initialisation

	bsf	PORTA,0		; turn on the led

; main loop

main:	clrf	COUNT		; clear ticks count
	clrf	TMR0		; clear internal clock count-up
A2:	bcf	INTCON,T0IF	; clear the TMR0 wrapped flag
A1:	btfss	INTCON,T0IF	; has TMR0 wrapped yet?
	goto	A3		; no, read the EEPROM
	incf	COUNT,F		; bump count of wraps
	movlw	10		; about 10 bumps/sec at 2MHz clock
B1:	xorwf	COUNT,W		; sets Z if COUNT = W 
B2:	btfss	STATUS,Z	; test Z
	goto	A2		; Z not set so COUNT != 10
	bcf	PORTA,0		; wraps COUNT = 10 so switch off led
	goto	main		; and repeat main loop



; code to read a byte from EEPROM

A3:	movlw	0		; EEPROM address 0
	movwf	EEADR		; into the address reg
	BANK1
	bsf	EECON1,RD	; initiate read
	BANK0
	movf	EEDATA,W	; copy EEPROM data to W
	movwf	EBYTE		; and store in main memory
	goto	A1		; go back and wait for TMR0 to wrap

; ----------------------------------------------------------
; Interrupt service routine
; ----------------------------------------------------------

ISR:	movwf	SAVEW		; save W 
	swapf	STATUS,W
	movwf	SAVES		; save STATUS
	BANK0			; ensure bank 0 set for PORT access

; first check that it's an RBIF interrupt 

	btfss	INTCON,RBIF	; test RBIF
	goto	POP		; not an RBIF interrupt

; read PORTB and compare with last PORTB value read
; see if either RB7 or RB6 have changed

	movf	PORTB,W		; read PORTB - also resets the input latches and clears mismatch
	movwf	BPORT		; and make a working copy in BPORT
	xorwf	SOFTB,W		; sets bits in W where changes from last value read
	andlw	b'11000000'	; are there changes in RB7 or RB6?
	btfsc	STATUS,Z	; Z clear if there are changes
	goto	CLR		; Z set so no changes - exit

; if RB7 or RB6 now reads 0, switch on the led
 
	btfss	BPORT,7		; skip if RB7 reads 1
	bsf	PORTA,0		; if RB7 = 0 switch on the led
	btfss	BPORT,6		; skip if RB6 reads 1
	bsf	PORTA,0		; if RB6 = 0 switch on the led

; housekeeping

CLR:	bcf	INTCON,RBIF	; clear the interrupt
	incf	ICOUNT,F	; bump count of interrupts
	movf	BPORT,W
	movwf	SOFTB		; and make an updated soft copy of PORTB

; restore context and exit

POP:	swapf   SAVES,W		; restore STATUS
	movwf	STATUS
	swapf	SAVEW,F		; restore W
	swapf	SAVEW,W
	retfie			; exit ISR

	end			; of program
