;*****************************************************************************        
;
;   Module:     hh.asm
;               
;   Author:     Mike Hibbett 
;                                                                  
;   Version:    1.2                                                 
;
;               Main source file for EPE 'Halloween Howler' project
;               This file includes all other source files.
;               
;               1.1  28/2/05 changed to 11025Hz
;               1.2  14/07/06 Corrected 3 input variant
;
;*****************************************************************************        

    list p=16f628, st=OFF, x=OFF, n=0
    errorlevel -302
    errorlevel -306

    #include <p16f628.inc>

    __config  0x3f26

    ; Pull in variable definitions and general constants
    #include "equates.inc"


;*****************************************************************************        
;
;   Function :  Reset vector
;               Hardware entry point to the code
;               This vector is also used for soft resets in the event of 
;               abnormal events
;
;   Input:      None.
;
;   Output:     N/A
;
;*****************************************************************************        
    
    ORG    0

RESET        GOTO    Main




;*****************************************************************************        
;
;   Function : uiWait100us
;              delays for a multiple of 100us
;
;   Input:     multiple in W
;
;*****************************************************************************        
uiWait100us
    movwf   delay2

d1us002
    movlw    D'165'                    
    movwf    delay1                   
    
d1us001    
    decfsz    delay1, F               
    goto    d1us001                    
    decfsz  delay2, F
    goto    d1us002
    return                         
    

    
;*****************************************************************************        
;
;   Function : uiWait10ms
;              delays for a multiple of 10ms
;
;   Input:     multiple in W
;
;*****************************************************************************        
uiWait10ms
    movwf   delay3
    
d1ms001
    clrwdt
    movlw   D'100'
    call    uiWait100us
    decfsz  delay3, F
    goto    d1ms001
    return



;*****************************************************************************        
;
;   Function :  initHardware
;               Basic hardware initialisation, suitable for 3 Input PCB setup
;               and for initial data download 
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
initHardware
    movlw   0x07
    movwf   CMCON
    bsf     STATUS,RP0              ;TRIS registers are in Bank1
    movlw   D2A_TRIS_VALUE          ; setup value - outputs
    movwf   D2A_TRIS                ; D2A outputs
    movlw   IO_TRIS_VALUE_3I        ; General I/O setup
    movwf   IO_TRIS
    bcf     STATUS,RP0              ;Restore Bank0
            
    return



;*****************************************************************************        
;
;   Function :  initLSRInputHW
;               Basic hardware initialisation for the LDR input PCB option
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
initLDRInputHW
    bsf     STATUS,RP0              ;TRIS registers are in Bank1
    movlw   IO_TRIS_VALUE           ; General I/O setup
    movwf   IO_TRIS
    bcf     STATUS,RP0              ;Restore Bank0
    return
        
        

;*****************************************************************************        
;
;   Function :  LoadE2Info
;               Reads the PCB and wav file information from the start of the
;               external eeprom. See i2c.inc for eeprom layout
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
LoadE2Info
    clrf    eepAddL
    clrf    eepAddH

    movlw   hwType
    movwf   FSR

le2i001
    call    eepRead
    movwf   INDF
    incf    eepAddL,F
    incf    FSR,F

    movfw   FSR
    sublw   wav3LENL + 1
    btfss   STATUS,Z
    goto    le2i001
    return
    

;*****************************************************************************        
;
;   Function :  windUp
;               slowly winds the D2A level from 0 to the value in W
;
;   Input:      start value in W
;
;   Output:     N/A
;
;*****************************************************************************        
windUp
    movwf   tmp2
    clrf    tmp1
    
wu001
    clrwdt
    movfw   tmp2
    subwf   tmp1,W
    btfsc   STATUS,Z
    return
    movfw   tmp1
    movwf   D2A_PORT
    movlw   0x0a
    call    uiWait100us
    
    incf    tmp1,F
    goto    wu001    
        
    
    
;*****************************************************************************        
;
;   Function :  windDown
;               slowly winds down the D2A level from current to 0
;
;   Input:      start value in W
;
;   Output:     N/A
;
;*****************************************************************************        
windDown
    movwf   tmp1
    
wd001
    clrwdt
    movfw   tmp1
    btfsc   STATUS,Z
    return
    movwf   D2A_PORT
    movlw   0x0a
    call    uiWait100us
    
    decf    tmp1,F
    goto    wd001    


;*****************************************************************************        
;
;   Function :  Main
;               Main application loop
;
;   Input:      None.
;
;   Output:     N/A
;
;*****************************************************************************        
Main
    clrf    STATUS                ; Select bank 0
    clrf    INTCON                ; No interrupts
    clrf    PCLATH                ; Code is in first bank

    ; Clear all bank0 ram
    movlw   0x20
    movwf   FSR
clrMem
    clrf    INDF
    incf    FSR, F
    btfss   FSR, 7              ; Have we reached 0x80?
    goto    clrMem              ; - No, so continue with clear

     ; infrequent wakes from wdt
    movlw   0xff
    bsf     STATUS,RP0           
    movwf   OPTION_REG
    bcf     STATUS,RP0           
    clrwdt

    call    initHardware  


    movlw   D'100'              ; powerup delay
    call    uiWait10ms          
    
    ; Quick RS232 check for data download request
    
    call    DownloadSound   
    
    ; Read the initial h/w setup from eeprom
    call    LoadE2Info
    
    movfw   hwType
    btfss   STATUS,Z
    goto    wait3Input      ; This never returns
    
    call    initLDRInputHW
    
    
; PCB Fitted with a single input, connected to an LDR
    
waitLDR2    
    ; turn the 'Eyes' off   
    movlw   0
    movwf   IO_PORT
            
    ; wait for low on LDR input
waitLDR    
    clrwdt
    movfw   IO_PORT
    andlw   0x10
    btfss   STATUS,Z
    goto    waitLDR
    
    ; turn the 'Eyes' on
    movlw   0x0c
    movwf   IO_PORT

    ; Setup 11025Hz interrupt interval in tmr2
    
    movlw   D'113'                   ; D'113' is 11.025KHz, with 1:4 prescale
    bsf     STATUS,RP0              ;TRIS registers are in Bank1
    movwf   PR2
    bcf     STATUS,RP0             
    
    movlw   0x05                    ; 1/4 prescale, 1/1 post
    movwf   T2CON

    movfw   wav1SAH
    movwf   eepAddH
    movfw   wav1SAL
    movwf   eepAddL

    clrf    dataCountH    
    clrf    dataCountL    
    
    call    eepReadFirst
    
    ; Slow increase the D2A output level to the first value
    call    windUp
    
wait    
    btfss   PIR1,1
    goto    wait
    
    bcf     PIR1,1

    call    eepReadNext
    movwf   D2A_PORT

    clrwdt
  
    incfsz  dataCountL,F
    goto    wait2
    incfsz  dataCountH,F
    
wait2  
    movfw   wav1LENH
    subwf   dataCountH,W
    btfss   STATUS,Z
    goto    wait3
    movfw   wav1LENL
    subwf   dataCountL,W
    btfss   STATUS,Z
    goto    wait3
    
    ; we have finished
    goto    endWav

wait3
    incfsz  eepAddL,F
    goto    wait
    incfsz  eepAddH,F
    goto    wait
    
endWav

    call    eepReadLast
    
    call    windDown        ; Slow reduce the D2A output level to 0
    
    goto    waitLDR2
    
    
    
    
; PCB fitted with 3 pulled up inputs    
wait3Input
    bsf     STATUS,RP0              ; TRIS is in bank 1
    movlw   0xFF
    movwf   TRISA
    bcf     STATUS,RP0              ;Restore Bank0
wait3Input1
    clrwdt
    movfw   IO_PORT
    andlw   0x1C
    sublw   0x1C
    btfsc   STATUS,Z
    goto    wait3Input1
    
    ; One of the inputs is low
    
    movfw   IO_PORT
    andlw   0x10
    btfsc   STATUS,Z
    goto    getwav1
    movfw   IO_PORT
    andlw   0x08
    btfsc   STATUS,Z
    goto    getwav2
    movfw   IO_PORT
    andlw   0x04
    btfsc   STATUS,Z
    goto    getwav3
    goto    wait3Input1 

getwav1
    movfw   wav1SAH
    movwf   wavSAH 
    movfw   wav1SAL
    movwf   wavSAL
    movfw   wav1LENH
    movwf   wavLENH 
    movfw   wav1LENL
    movwf   wavLENL 
    goto    playwav

getwav2
    movfw   wav2SAH
    movwf   wavSAH 
    movfw   wav2SAL
    movwf   wavSAL
    movfw   wav2LENH
    movwf   wavLENH 
    movfw   wav2LENL
    movwf   wavLENL 
    goto    playwav
    
getwav3
    movfw   wav3SAH
    movwf   wavSAH 
    movfw   wav3SAL
    movwf   wavSAL
    movfw   wav3LENH
    movwf   wavLENH 
    movfw   wav3LENL
    movwf   wavLENL 
    
playwav
    ; Setup 11025Hz interrupt interval in tmr2
    
    movlw   D'113'                   ; D'113' is 11.025KHz, with 1:4 prescale
    bsf     STATUS,RP0              ;TRIS registers are in Bank1
    movwf   PR2
    bcf     STATUS,RP0             
    
    movlw   0x05                    ; 1/4 prescale, 1/1 post
    movwf   T2CON

    movfw   wavSAH
    movwf   eepAddH
    movfw   wavSAL
    movwf   eepAddL

    clrf    dataCountH    
    clrf    dataCountL    
    
    call    eepReadFirst
    
    ; Slow increase the D2A output level to the first value
    call    windUp
    
waita    
    btfss   PIR1,1
    goto    waita
    
    bcf     PIR1,1

    call    eepReadNext
    movwf   D2A_PORT

    clrwdt
  
    incfsz  dataCountL,F
    goto    waitb
    incfsz  dataCountH,F
    
waitb
    movfw   wavLENH
    subwf   dataCountH,W
    btfss   STATUS,Z
    goto    waitc
    movfw   wavLENL
    subwf   dataCountL,W
    btfss   STATUS,Z
    goto    waitc
    
    ; we have finished
    goto    endpWav

waitc
    incfsz  eepAddL,F
    goto    waita
    incfsz  eepAddH,F
    goto    waita
    
endpWav

    call    eepReadLast
    
    call    windDown        ; Slow reduce the D2A output level to 0
    

    ; wait for input to go high
wait3Inputh
    clrwdt
    movfw   IO_PORT
    andlw   0x1C
    sublw   0x1C
    btfss   STATUS,Z
    goto    wait3Inputh

    goto    wait3Input1
        
    
    
;*****************************************************************************        
;
;   Function :  SerialReceive
;               This function extracts a byte from the receiver hardware.
;               It will block until a byte is available 
;
;   Input:      None.
;
;   Output:     byte in W
;
;*****************************************************************************        
SerialReceive
    clrwdt
    btfss   PIR1,RCIF    ;check if data received
    goto    SerialReceive            ;wait until new data
    movf    RCREG,W        ;get received data into W
    return



;*****************************************************************************        
;
;   Function :  SerialTransmit
;               This function sends the byte in W over the RS232 port. The 
;               function will wait until previous data has been sent
;
;   Input:      Byte in W
;
;   Output:     
;
;*****************************************************************************        
SerialTransmit
    clrwdt
    btfss   PIR1,TXIF    ;check that buffer is empty
    goto    SerialTransmit
    movwf   TXREG        ;transmit byte
    return
    
    
;*****************************************************************************        
;
;   Function :  hex2byte
;               This function converts the ascii hex digits in rxB1 and rxB2 
;               into W, as a single byte. (rxB1 MSB)                           
;               Alpha digits are uppercase
;   Input:               
;
;   Output:     
;
;      Uses:    tmp1
;
;*****************************************************************************        
hex2byte     
    movfw   rxB1
    sublw   0x39
    movfw   rxB1
    andlw   0x0F
    btfss   STATUS, C
    addlw   0x09
    movwf   tmp1
    swapf   tmp1,F
    movfw   rxB2
    sublw   0x39
    movfw   rxB2
    andlw   0x0F
    btfss   STATUS, C
    addlw   0x09
    addwf   tmp1,W
    return



;*****************************************************************************        
;
;   Function :  byte2hex
;               This function converts the byte W into two hex characters
;               rxB1 and rxB2 (rxB1 MSB)                            
;               Alpha digits are uppercase
;   Input:               
;
;   Output:     
;
;      Uses:    tmp1
;
;*****************************************************************************        
byte2hex     
    movwf   tmp1
    swapf   tmp1,W
    sublw   0x09
    swapf   tmp1,W
    andlw   0x0F
    btfss   STATUS,DC
    addlw   'A' - .10 - '0'
    addlw   '0'
    movwf   rxB1
    movfw   tmp1
    sublw   0x09
    movfw   tmp1
    andlw   0x0F
    btfss   STATUS,DC
    addlw   'A' - .10 - '0'
    addlw   '0'
    movwf   rxB2
    return


;*****************************************************************************        
;
;   Function :  DownloadSound
;               This function polls a PC host then looks for an ack.
;               If it gets an ack, it then downloads data to the eeprom
;
;   Input:               
;
;   Output:     
;
;*****************************************************************************        
DownloadSound
    ; Setup I/O for RS232 comms
    bsf     STATUS,RP0              ;TRIS registers are in Bank1
    movlw   RS232_TRIS_VALUE        ; setup value - outputs, except RX/TX
    movwf   D2A_TRIS                ; D2A outputs
    bcf     STATUS,RP0              ;Restore Bank0

    ; Setup RS232 Port for polled I/O
    bsf     STATUS,RP0    
    movlw   BAUD_CONSTANT           ;set baud rate 
    movwf   SPBRG
    bsf     TXSTA,BRGH              ;baud rate high speed option
    bsf     TXSTA,TXEN              ;enable transmission
    bcf     STATUS,RP0    
    bsf     RCSTA,CREN              ;enable reception
    bsf     RCSTA,SPEN              ;enable serial port
    
    movf    RCREG, W                ; Clear any false rx'ed data
    bcf     PIR1, RCIF
    
    call    eepHWInit
    
    ; Signal to the PC that we are ready
    movlw   'A'
    call    SerialTransmit    

    ; Use eepData as a simple retry counter
    movlw   0xff
    movwf   eepData
    
ds000
    btfsc   PIR1,RCIF    ;check if data received
    goto    gotByte

    ; Wait, then look again, up to 
    movlw   0x01
    call    uiWait10ms
    decfsz  eepData,F
    goto    ds000
    goto    dsexit
        
gotByte
    movf    RCREG,W        ;get received data into W
    movwf   tmp1
    clrf    eepAddL
    clrf    eepAddH

    sublw   'A'
    btfss   STATUS,Z
    goto    trydumpE2

gb000    
    call    SerialReceive
    movwf   rxB1
    movfw   rxB1
    call    SerialTransmit
    call    SerialReceive
    movwf   rxB2

    ; Convert rxB1 / rxB2 to a single byte, in eepData
    call    hex2byte
    movwf   eepData
    
    movfw   eepAddL
    andlw   0x7F
    sublw   0x00
    btfss   STATUS,Z
    goto    gb003       ; Not first
    
    call    eepWritePageFirst
    goto    gb002
    
gb003    
    movfw   eepAddL
    andlw   0x7F
    sublw   0x7F
    btfss   STATUS,Z
    goto    gb001       ; Page write    
    
    call    eepWritePageStop

    ; 5ms write time, max
    movlw   D'50'
    call    uiWait100us
    goto    gb002
    
gb001
    call    eepWritePageNext

gb002
    movfw   rxB2
    call    SerialTransmit
    incfsz  eepAddL,F
    goto    gb000
    incfsz  eepAddH,F
    goto    gb000

    ; Restore the working defaults
    bsf     STATUS,RP0              ;TRIS registers are in Bank1
    movlw   D2A_TRIS_VALUE          ; setup value - outputs
    movwf   D2A_TRIS                ; D2A outputs
    bcf     STATUS,RP0              ;Restore Bank0
    
    goto    dsexit
    
trydumpE2
    movfw   tmp1
    sublw   'D'    
    btfss   STATUS,Z
    goto    ds000
    
dumpE2    
    call    eepReadFirst
    call    byte2hex
    movfw   rxB1
    call    SerialTransmit
    movlw   0x01
    call    uiWait10ms
    movfw   rxB2
    call    SerialTransmit
    movlw   0x01
    call    uiWait10ms

    incfsz  eepAddL,F
    goto    de001
    incf    eepAddH,F

de001

    call    eepReadNext
    call    byte2hex
    movfw   rxB1
    call    SerialTransmit
    movlw   0x01
    call    uiWait10ms
    movfw   rxB2
    call    SerialTransmit
    movlw   0x01
    call    uiWait10ms

    incfsz  eepAddL,F
    goto    de001
    incf    eepAddH,F
    
    btfss   eepAddH,5       ; 8KB ; TODO Should really be ,7, for 32KB
    goto    de001
    
    ; Dummy read to complete the transmission
    call    eepReadLast

dsexit    
    ; Restore the working defaults
    bsf     STATUS,RP0              ;TRIS registers are in Bank1
    movlw   D2A_TRIS_VALUE          ; setup value - outputs
    movwf   D2A_TRIS                ; D2A outputs
    bcf     STATUS,RP0              ;Restore Bank0

    bcf     RCSTA,SPEN              ;disable the serial port
    
    return    

    ; Include the source code for the eeprom control
    #include "i2c.inc"

    
    END                ; End of program
    