; Icebox program for PIC12F675
; This version does different led flash patterns
; GP2 is AD2 input
; red led on GP4
; green led on GP5
; internal (4MHz) oscillator

	list	p=12f675,r=dec

; config int osc, GP4 is I/O, PWRTE, WDTE

	__config	h'01AC'

; Macros

; Remove ';' comment out to do OSCCAL calibration
; #define DO_OSCCAL

#define BANK0   BCF 0x03,5
#define BANK1   BSF 0x03,5

#define SPEED		18	; basic led flash rate (higher no = slower)  20 = 1.6 Hz
#define ADRATE		4	; times round main loop between polls of the temp sensor
#define	SLOWRATE	3	; slow flash rate * basic rate
#define FASTRATE	1	; fast flash rate * basic rate

#define GREENLED 5		; bit positions in GPIO
#define REDLED   4

; A/D thresholds for different flash rates
; 220 (op-amp gain) * 0.0025v/C (diode temp characteristic) = 0.55v/C
; 256 a/d = 5v:  1 C = 0.55v = 256*0.55/5 = 28.2 a/d 
; intervals below are 30 or 31 a/d wide.  30.5/28.16 = 1.08
; so each interval is a little less than 1.1 C
; SOLIDGREEN is about 2.98/0.55 = 5.4 C above REDGREENFLASH
; TL082 output swings between about 0.7v and 4.3v
 
#define	SOLIDGREEN	204	; 3.98v
#define GREENSLOWFLASH	173	; 3.38v
#define	GREENFASTFLASH	142	; 2.77v
#define SOLIDRED	112	; 2.19v
#define REDSLOWFLASH	81	; 1.58v
#define	REDFASTFLASH	51	; 1.00v
; REDGREENFLASH below this
 
; Equates for registers

	include	p12f675.inc

; Data locations (start at 0x20)

MLOOPS	equ	0x20	; counts no times round main loop between A/D polls
THERM	equ	0x21	; last thermometer reading /4
COUNT	equ	0x22	; count of timer ticks
TOGVAL	equ	0x23	; led flash XOR toggle value
FRATE	equ	0x24	; flash rate save value
FRATECT	equ	0x25	; flash rate counter

; code
	org	0
	goto	init
	nop
	nop
	nop
	retfie		; interrupt vector

init:	

; interrupt setup

	bcf	INTCON,GIE	; disable global interrupts
	clrwdt			; clear watchdog timer

; initialise PIC

	clrf	GPIO		; initialise all port outputs to zero
	movlw	0x07
	movwf	CMCON		; turn off comparator
	BANK1
#ifdef DO_OSCCAL
	call	0x3FF
	movwf	OSCCAL		; calibrate internal osc
#endif
	movlw	b'00001111'	; GP4/5 output, 0-3 input
	movwf	TRISIO
	movlw	b'00010100'	; Tosc/8, set AN2
	movwf	ANSEL
	movlw	b'00000101'	; enable GPIO pullups, PSA to TMR, prescale /64
	movwf	OPTION_REG
	BANK0
	movlw	b'11001001'	; right j, vdd is ref, select AN2, ADON
	movwf	ADCON0

; data initialisation

	movlw	1		; do first A/D straight away
	movwf	MLOOPS
	clrf	TOGVAL

; main loop

main
	clrf	TMR0		; clear internal clock count-up
	clrf	COUNT		; clear ticks count
A2:	bcf	INTCON,T0IF	; clear the TMR0 wrapped flag

A1:	clrwdt			; reset watchdog timer
	btfss	INTCON,T0IF	; has TMR0 wrapped yet?
	goto	A1		; no, loop till it has
	incf	COUNT,F		; bump count of wraps
	movlw	SPEED
	xorwf	COUNT,W		; sets Z if COUNT = W 
	btfss	STATUS,Z	; test Z
	goto	A2		; Z not set so COUNT != SPEED

#ifdef NEVER
	movlw	0x30		; wraps COUNT = SPEED so
	xorwf	GPIO,F		; toggle GPIO 4 and 5
	goto	A2
#endif

	decfsz	MLOOPS,F	; time for an A/D?
	goto	flash		; no

; read thermometer

	bsf	ADCON0,GO	; start A/D
	movlw	ADRATE
	movwf	MLOOPS		; reset MLOOPS
	btfsc	ADCON0,GO	; wait for A/D to complete
	goto	$-1

; result is 10-bit A/D in ADRESH and ADRESL
; lose the least 2 sig bits

	clrwdt
	rrf	ADRESH,F	; ls bit of ADRESH to C
	BANK1
	rrf	ADRESL,F	; C to ms bit of ADRESL, ls bit lost
	BANK0
	rrf	ADRESH,F
	BANK1
	rrf	ADRESL,W
	BANK0
	movwf	THERM		; save thermometer reading

; set up flash according to THERM value

	movlw	SOLIDGREEN	; threshold
	subwf	THERM,W		; subtract from value
	btfsc	STATUS,C	; C is set if there was no borrow
	goto	initgreen	; C set so THERM > test value

	movlw	GREENSLOWFLASH	; threshold
	subwf	THERM,W		; subtract from value
	btfsc	STATUS,C	; C is set if there was no borrow
	goto	initgrsf	; C set so THERM > test value

	movlw	GREENFASTFLASH	; threshold
	subwf	THERM,W		; subtract from value
	btfsc	STATUS,C	; C is set if there was no borrow
	goto	initgrff	; C set so THERM > test value

	movlw	SOLIDRED	; threshold
	subwf	THERM,W		; subtract from value
	btfsc	STATUS,C	; C is set if there was no borrow
	goto	initred		; C set so THERM > test value

	movlw	REDSLOWFLASH	; threshold
	subwf	THERM,W		; subtract from value
	btfsc	STATUS,C	; C is set if there was no borrow
	goto	initrsf		; C set so THERM > test value

	movlw	REDFASTFLASH	; threshold
	subwf	THERM,W		; subtract from value
	btfsc	STATUS,C	; C is set if there was no borrow
	goto	initrff		; C set so THERM > test value
	
	goto	initgrfl	; lowest is green/red flash
	
; the following routines set up the display and flash parameters

initgreen			; green solid

	bsf	GPIO,GREENLED	
	bcf	GPIO,REDLED
	clrf	TOGVAL		; no change
				; rates don't matter because no change
	goto	main

initgrsf			; green slow flash

	bsf	GPIO,GREENLED
	bcf	GPIO,REDLED
	bsf	TOGVAL,GREENLED	; toggle green led
	bcf	TOGVAL,REDLED
	movlw	SLOWRATE
	movwf	FRATE
	movwf	FRATECT
	goto	main

initgrff			; green fast flash

	bsf	GPIO,GREENLED
	bcf	GPIO,REDLED
	bsf	TOGVAL,GREENLED
	bcf	TOGVAL,REDLED
	movlw	FASTRATE
	movwf	FRATE
	movwf	FRATECT
	goto	main

initred				; red solid

	bsf	GPIO,REDLED	
	bcf	GPIO,GREENLED
	clrf	TOGVAL
	goto	main

initrsf				; red slow flash

	bsf	GPIO,REDLED
	bcf	GPIO,GREENLED
	bsf	TOGVAL,REDLED
	bcf	TOGVAL,GREENLED
	movlw	SLOWRATE
	movwf	FRATE
	movwf	FRATECT
	goto	main

initrff				; red fast flash
	bsf	GPIO,REDLED
	bcf	GPIO,GREENLED
	bsf	TOGVAL,REDLED
	bcf	TOGVAL,GREENLED
	movlw	FASTRATE
	movwf	FRATE
	movwf	FRATECT
	goto	main

initgrfl			; green/red flash

	bsf	GPIO,GREENLED
	bcf	GPIO,REDLED
	bsf	TOGVAL,GREENLED
	bsf	TOGVAL,REDLED
	movlw	FASTRATE
	movwf	FRATE
	movwf	FRATECT
	goto	main


; main loop: flash the led according to last read value

flash
	decfsz	FRATECT,F
	goto	main
	movf	FRATE,W
	movwf	FRATECT	
	movf	TOGVAL,W	; toggle value
	xorwf	GPIO,F		; toggle led display
	goto	main		; and repeat main loop

        end			; of program


