;*****************************************************************************        
;
;   Module:     melody.inc
;               
;   Author:     Mike Hibbett 
;                                                                  
;   Version:    0.1 29/06/05                                                  
;
;               This is the 'tune' module, which holds the note frequencies,
;               tunes and code to support them.
;               The tunes are played by simple bit toggle of an output pin
;               using timer0 as the period timer.
;               The more adventurous of y9ou may consider implementing a second
;               note generator, using one of the other timers, to generate
;               two note melodies. This is all a bit over the top but hey, it's
;               fun!
;
;
;*****************************************************************************        

; Tune generation is started by writing a tune number ( 0... FF )
; into tuneCurrent.
; The currently play tune can, if required, be turned off by writing 0
; into tuneCurrent.
; The interrupt routine uses intTuneCurrent, intTunePosition and intNoteTimer
; to manage the playing of a tune.




;*****************************************************************************        
;
;   Function :  MelodyMaker
;               'state machine' for tune generation and halting
;
;   Input:      None. Called every 50ms
;
;   Output:     None, as such
;               NOTE: For speed this routine is jumped to rather than called,
;               therefore it does an interrupt return 
;
;*****************************************************************************        

MelodyMaker
    ; have we been asked to play a different tune?
    movf    tuneCurrent,W
    subwf   intTuneCurrent, W
    bz      MM001
    
    tstfsz  tuneCurrent
    bra     MM000
    
    ; Request to cancel current tune
    
    clrf    intTuneCurrent
    
    clrf    T0CON   
    bcf     INTCON, TMR0IF
    bcf     LATC, 2     ; sound pin off, no current drain

    retfie  FAST
       
MM000    
    ; New tune requested
    
    movf    TBLPTRL, W  
    movwf   intTBLPTRL              
    movf    TBLPTRH, W  
    movwf   intTBLPTRH              
    movf    TABLAT, W  
    movwf   intTABLAT          
    
    movf    tuneCurrent, W
    movwf   intTuneCurrent
    bra     MM004
    
MM001    
    tstfsz  intTuneCurrent              ; A tune of zero means none
    bra     MM002                       ; .. Non zero, we are playing a tune
    retfie  FAST                        ; .. So just exit  

MM002
    ; playing a tune. Deal with it...
    
    ; decrement note timer. If zero, move to next note
    decfsz  intNoteTimer, F
    retfie  FAST                        ; Thats all, just keep playing this note
    
    ; This note has ended - next note
    ; that means incrementing note index, getting the note value into reload store, getting duration into timer
    ; it may mean repeating from beginning, and it may mean stopping, it may mean being quiet for a bit.
    
    movf    TBLPTRL, W  
    movwf   intTBLPTRL              
    movf    TBLPTRH, W  
    movwf   intTBLPTRH              
    movf    TABLAT, W  
    movwf   intTABLAT          
    
    ; next note
    infsnz  intNotePtrL, F
    incf    intNotePtrH, F

MM0022    
    movf    intNotePtrL, W
    movwf   TBLPTRL
    movf    intNotePtrH, W
    movwf   TBLPTRH
    tblrd*+                     ; Read note duration
    movf    TABLAT, W
    movwf   intNoteTimer

    infsnz  intNotePtrL, F
    incf    intNotePtrH, F

    tblrd*+                     ; Read note number
    movf    TABLAT, W
    
    sublw   0xFF
    bz      MM003                ; END
    
    movf    TABLAT, W
    sublw   0xFE
    bz      MM004                ; REPEAT TUNE ad nauseium
    
    movf    TABLAT, W
    sublw   0xFD
    bz      MM005                ; Silent note
    
    ; Normal note!
    movlw   0x80                 ; 10000000   1/2 i clock rate
    movwf   T0CON
    bsf     INTCON, TMR0IF      ; Just in case last note was silent
    
    ; get note data...
    movlw   HIGH NOTES
    movwf   TBLPTRH
    movlw   LOW NOTES
    movwf   TBLPTRL
    
    movf    TABLAT, W           ; Note number
    addwf   TABLAT, W           ; Create a word index, ie * 2
    
    addwf   TBLPTRL, F
    movlw   0x00
    addwfc  TBLPTRH, F
    
    tblrd*+                     ; Read note high value
    movf    TABLAT, W
    movwf   tmr0ValH
    tblrd*+                     ; Read note low value
    movf    TABLAT, W
    movwf   tmr0ValL
    bra     MM006    

MM004
    ; Repeat from beginning

    movlw   HIGH TuneList
    movwf   TBLPTRH
    movlw   LOW TuneList
    movwf   TBLPTRL
    
    movf    intTuneCurrent, W
    addwf   intTuneCurrent, W   ; Create a word index, ie * 2

    addwf   TBLPTRL, F
    movlw   0x00
    addwfc  TBLPTRH, F
    
    tblrd*+                     ; Read tune address high value
    movf    TABLAT, W
    movwf   intNotePtrL
    tblrd*+                     ; Read tune address low value
    movf    TABLAT, W
    movwf   intNotePtrH

    bra     MM0022              ; Go back and play the first note

MM003
    ; End tune playing
    clrf    intTuneCurrent
    clrf    tuneCurrent
    ; fall through to next case, and re-use it
    
MM005
    ; Silent note
    clrf    T0CON   
    bcf     INTCON, TMR0IF
    bcf     LATC, 2     ; sound pin off, no current drain

MM006
    ; Restore interrupted registers, and exit    
    movf    intTBLPTRL, W  
    movwf   TBLPTRL              
    movf    intTBLPTRH, W  
    movwf   TBLPTRH              
    movf    intTABLAT, W  
    movwf   TABLAT          
        
    retfie  FAST


; TuneList is used as a simple-to-index-into list, and is what the application uses when
; writing into tuneCurrent.
TuneList
    DW      0       ; A tune index of zero means 'no tune'. This just helps the coding
    DW      TUNE1
    DW      TUNE2
    DW      TUNE3   
    DW      TUNE4
    DW      TUNE5
    DW      TUNE6
    DW      TUNE9
    DW      TUNE8
    
    
; The actual tunes follow. The rules for generating tunes are thus:
; 1) Have a musical ear. Amazing are the horrors that the inept can achieve :o)
; 2) A tune consists of byte pairs; DURATION:NOTE.  A NOTE of FD is silence, FE is repeat. FF is end
;    DURATION is in 50ms intervals
    
TUNE1   ; Power up jingle
    DB 0x01,0x0C, 0x01,0x0E, 0x01,0x10, 0x01,0x11, 0x01,0x13, 0x01,0x15, 0x01,0x17, 0x01,0x18
    DB 0x01,0x18, 0x01,0x17, 0x01,0x15, 0x01,0x13, 0x01,0x11, 0x01,0x10, 0x01,0x0E, 0x01,0x0C, 0x00,0xFF

TUNE2   ; GPS Lock occurs
    DB 0x02,0x27, 0x02,0xFD, 0x02,0x27, 0x00,0xFF
    
TUNE3   ; GPS Lock lost
    DB 0x06,0x0C, 0x00,0xFF

TUNE4
    ; Police siren?
    DB 0x06,0x13, 0x06,0x17
    DB 0x00,0xFE

TUNE5
    ; slow beep
    DB 0x02,0x18, 0x28, 0xFD
    DB 0x00,0xFE

TUNE6
    ; fast beep
    DB 0x02,0x18, 0x10, 0xFD
    DB 0x00,0xFE

TUNE7
    ; continuous tone
    DB 0xff,0x18, 0x00,0xFE


TUNE8
    ; quick beep, keypress
    DB 0x02,0x27, 0x00,0xFF


TUNE9
    ; very fast beep
    DB 0x02,0x18, 0x02, 0xFD
    DB 0x00,0xFE
    
    
; 48 notes, 4 octaves. These are reload values into timer0. 220Hz to 3878Hz. Best to avoid changing these
; unless you know what you are doing; see point 1) above.
NOTES
    DB  0xB3, 0xAA      ; 0x00 Middle C
    DB  0xB7, 0xEB      ; 0x01 C#
    DB  0xBB, 0xEF      ; 0x02 D 
    DB  0xBF, 0xBA      ; 0x03 D#
    DB  0xC3, 0x4E      ; 0x04 E 
    DB  0xC6, 0xB0      ; 0x05 F 
    DB  0xC9, 0xE1      ; 0x06 F#
    DB  0xCC, 0xE5      ; 0x07 G 
    DB  0xCF, 0xBE      ; 0x08 G#
    DB  0xD2, 0x6E      ; 0x09 A 
    DB  0xD4, 0xF8      ; 0x0A A#
    DB  0xD7, 0x5E      ; 0x0B B 
    DB  0xD9, 0xA2      ; 0x0C C
    DB  0xDB, 0xC5      ; 0x0D C#
    DB  0xDD, 0xCA      ; 0x0E D 
    DB  0xDF, 0xB2      ; 0x0F D#
    DB  0xE1, 0x7E      ;  0x10 E 
    DB  0xE3, 0x31      ; 0x11 F 
    DB  0xE4, 0xCC      ; 0x12 F#
    DB  0xE6, 0x50      ;  0x13 G 
    DB  0xE7, 0xBF      ; 0x14 G#
    DB  0xE9, 0x19      ;  0x15 A 
    DB  0xEA, 0x5F      ; 0x16 A#
    DB  0xEB, 0x94      ;  0x17 B 
    DB  0xEC, 0xB7      ; 0x18 C
    DB  0xED, 0xCA      ; 0x19 C#
    DB  0xEE, 0xCE      ; 0x1A D 
    DB  0xEF, 0xC3      ; 0x1B D#
    DB  0xF0, 0xAB      ; 0x1C E 
    DB  0xF1, 0x85      ; 0x1D F 
    DB  0xF2, 0x54      ; 0x1E F#
    DB  0xF3, 0x17      ; 0x1F G 
    DB  0xF3, 0xCF      ; 0x21 G#
    DB  0xF4, 0x7D      ; 0x22 A 
    DB  0xF5, 0x21      ; 0x23 A#
    DB  0xF5, 0xBC      ; 0x24 B 
    DB  0xF6, 0x4F      ; 0x25 C
    DB  0xF6, 0xD9      ; 0x26 C#
    DB  0xF7, 0x5B      ; 0x27 D 
    DB  0xF7, 0xD7      ; 0x28 D#
    DB  0xF8, 0x4B      ; 0x29 E 
    DB  0xF8, 0xB9      ; 0x2A F 
    DB  0xF9, 0x21      ; 0x2B F#
    DB  0xF9, 0x83      ; 0x2C G 
    DB  0xF9, 0xDF      ; 0x2D G#
    DB  0xFA, 0x37      ; 0x2E A 
    DB  0xFA, 0x89      ; 0x2F A#
    DB  0xFA, 0xD7      ; 0x30 B 
                