;*****************************************************************************        
;
;   Module:     spieeprom.inc
;               
;   Author:     Mike Hibbett 
;                                                                  
;   Version:    0.1 27/06/05                                            
;
;               Functions to communicate with two 25LC256'sover the SPI bus
;
;*****************************************************************************        


;*****************************************************************************        
;
;   EEPROM Memory Map
;
;   The data stored in the eeprom is defined below
;
;
;    Address          Description                Size
;    (Hex)                                      (Bytes)   
;    0000    Magic Number ( 0x55, 0xAA )         2
;    0002    Number of records stored            2
;    0004    Default Display Number              1
;    0005    spare ( to make the maths easy)     3
;    0008    Records                             n * 6
;            format is
;            binary minutes, signed lat          3
;            binary minutes, signed long         3
;            Records are stored as 24 bit signed
;            numbers. Latitude then longitude.
;            The units are in 1/100 minute, giving about 18M resolution
;            ie 10degreesE is stored as 60000,
;            therefore 0x00EA60 in eeprom.
;         
;            No more than 10,800 camera positions will be stored; this allows 
;            for some free space at the end of the eeprom for future use
;
;   Note: Magic number is used to check that the eeprom has
;         been configured.   
;         All multibyte values are stored MSB first
;         **NOTE**: If you change the layout of memory map, ensure that 
;         the records start at an address that does not cause a record
;         to span the two eeproms. The code expects to be able to read 6 
;         consecutive bytes without needing to change the chip selects 
;
;*****************************************************************************        



;*****************************************************************************        
;
;   Function :  EEPHWInit
;               configures the SPI peripherial hardware
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
EEPHWInit
    ; We will use Mode (1,1)
    
    movlw   SSPSTAT_VAL
    movwf   SSPSTAT
    
    movlw   SSPCON1_VAL
    movwf   SSPCON1
    return



;*****************************************************************************        
;
;   Function :  EEPCheck
;               If the eeprom is uninitialised, calls eepClear
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
EEPCheck
    clrf    eepAddH
    clrf    eepAddL
    call    EEPRead

    movf    eepData, W
    sublw   0x55
    btfss   STATUS, Z
    goto    eepc_001
    
    incf    eepAddL, F
    call    EEPRead

    movf    eepData, W
    sublw   0xAA
    btfss   STATUS, Z
    goto    eepc_001
    return
    
eepc_001
    call    EEPClear

    ; Display a message explaining that the eeprom is/has being erased

    call    DspHome
    call    DspClear

    DISPLAY STREEPErase
    call    DspLine2   
    
    movlw   D'250'
    call    UIWait10ms
        
    return



;*****************************************************************************        
;
;   Function :  EEPClear
;               Clears the eeprom array ( well, the important parts anyway )
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
EEPClear
    movlw   0x55
    movwf   eepData
    clrf    eepAddH
    clrf    eepAddL
    call    EEPWrite            ; Magic number 1
    movlw   0xAA
    movwf   eepData
    call    EEPWrite            ; Magic number 2
    clrf    eepData
    call    EEPWrite            ; Number of records
    call    EEPWrite
    call    EEPWrite            ; Default display number
    return



;*****************************************************************************        
;
;   Function :  EEPWrite
;               This is a global function for writing data to the eeprom
;
;   Input:      EEPROM address in eepAddH + eepAddL. 
;               
;
;   Output:     Byte to send in eepData. Input address is incremented by 1
;
;*****************************************************************************        
EEPWrite
    ; Enable appropriate chip select
    btfsc   eepAddH, 7
    bcf     LATC, 1
    
    btfss   eepAddH, 7
    bcf     LATC, 0
    
    nop
    nop
    
    movlw   EEPROM_WREN
    movwf   SSPBUF              ; Transmit the command byte

EEP001
    btfss   SSPSTAT, BF
    bra     EEP001
    
    movf    SSPBUF, W           ; Dummy read

    bsf     LATC, 0
    bsf     LATC, 1

    nop
    nop

    ; Enable appropriate chip select
    btfsc   eepAddH, 7
    bcf     LATC, 1
    
    btfss   eepAddH, 7
    bcf     LATC, 0
    
    nop
    nop


    movlw   EEPROM_WRITE
    movwf   SSPBUF              ; Transmit the command byte

EEP0011
    btfss   SSPSTAT, BF
    bra     EEP0011
    
    movf    SSPBUF, W           ; Dummy read


    movf    eepAddH, W
    movwf   SSPBUF              ; Transmit the address byte

EEP002
    btfss   SSPSTAT, BF
    bra     EEP002
    
    movf    SSPBUF, W           ; Dummy read

    movf    eepAddL, W
    movwf   SSPBUF              ; Transmit the address byte

EEP003
    btfss   SSPSTAT, BF
    bra     EEP003
    
    movf    SSPBUF, W           ; Dummy read
    
          
    movf    eepData, W
    movwf   SSPBUF              ; Transmit the data byte

EEP004
    btfss   SSPSTAT, BF
    bra     EEP004
    
    movf    SSPBUF, W           ; Dummy read
    movwf   eepData
    
    bsf     LATC, 0
    bsf     LATC, 1

    ; Allow eeprom time to write the byte
    movlw   0x01                
    call    UIWait10ms          
    
    ; increment the eeprom address pointer
    movlw   0x01                    
    addwf   eepAddL,F
    btfsc   STATUS,C
    incf    eepAddH,F
    
    return



;*****************************************************************************        
;
;   Function :  EEPRead
;               This is a global function for reading data from the eeprom
;
;   Input:      EEPROM address in eepAddH + eepAddL. 
;               
;
;   Output:     Received byte in eepData
;
;*****************************************************************************        
EEPRead
    ; Enable appropriate chip select
    btfsc   eepAddH, 7
    bcf     LATC, 1
    
    btfss   eepAddH, 7
    bcf     LATC, 0
    
    nop
    nop
    
    movlw   EEPROM_READ
    movwf   SSPBUF              ; Transmit the command byte

EER001
    btfss   SSPSTAT, BF
    bra     EER001
    
    movf    SSPBUF, W           ; Dummy read

    movf    eepAddH, W
    movwf   SSPBUF              ; Transmit the address byte

EER002
    btfss   SSPSTAT, BF
    bra     EER002
    
    movf    SSPBUF, W           ; Dummy read

    movf    eepAddL, W
    movwf   SSPBUF              ; Transmit the address byte

EER003
    btfss   SSPSTAT, BF
    bra     EER003
    
    movf    SSPBUF, W           ; dummy byte
 
    movf    eepAddL, W
    movwf   SSPBUF              ; Transmit the address byte

EER004
    btfss   SSPSTAT, BF
    bra     EER004
    
    movf    SSPBUF, W           ; read byte
    movwf   eepData
   
    bsf     LATC, 0
    bsf     LATC, 1
    return



;*****************************************************************************        
;
;   Function :  EEPReadPosition
;               This is a global function, specially created for the camerawatch 
;               project. It reads 6 consecutive locations
;               storing the data into scanLat and scanLong
;
;   Input:      EEPROM address in eepAddH + eepAddL. 
;               
;
;   Output:     6 bytes in scanLat and scanLong
;
;*****************************************************************************        
;EEPReadPosition
;    call    EEPRead
;    movf    eepData, W
;    movwf   scanLat
    
;    infsnz  eepAddL, F
;    incf    eepAddH, F
;    call    EEPRead
;    movf    eepData, W
;    movwf   scanLat+1
    
;    infsnz  eepAddL, F
;    incf    eepAddH, F
;    call    EEPRead
;    movf    eepData, W
;    movwf   scanLat+2
    
;    infsnz  eepAddL, F
;    incf    eepAddH, F
;    call    EEPRead
;    movf    eepData, W
;    movwf   scanLong
    
;    infsnz  eepAddL, F
;    incf    eepAddH, F
;    call    EEPRead
;    movf    eepData, W
;    movwf   scanLong+1
    
;    infsnz  eepAddL, F
;    incf    eepAddH, F
;    call    EEPRead
;    movf    eepData, W
;    movwf   scanLong+2
;    return




;*****************************************************************************        
;
;   Function :  EEPReadPositionFast
;               This is a global function, specially created for the camerawatch 
;               project. It reads 6 consecutive locations  as fast as possible,
;               storing the data into scanLat and scanLong
;
;   Input:      EEPROM address in eepAddH + eepAddL. 
;               
;
;   Output:     6 bytes in scanLat and scanLong
;
;*****************************************************************************        
EEPReadPositionFast
    ; Enable appropriate chip select
    btfsc   eepAddH, 7
    bcf     LATC, 1
    
    btfss   eepAddH, 7
    bcf     LATC, 0
    
    nop
    nop
    
    movlw   EEPROM_READ
    movwf   SSPBUF              ; Transmit the command byte

EERF001
    btfss   SSPSTAT, BF
    bra     EERF001
    
    movf    SSPBUF, W           ; Dummy read

    movf    eepAddH, W
    movwf   SSPBUF              ; Transmit the address byte

EERF002
    btfss   SSPSTAT, BF
    bra     EERF002
    
    movf    SSPBUF, W           ; Dummy read

    movf    eepAddL, W
    movwf   SSPBUF              ; Transmit the address byte

EERF003
    btfss   SSPSTAT, BF
    bra     EERF003
    
    movf    SSPBUF, W           ; dummy byte
 
    movf    eepAddL, W
    movwf   SSPBUF              ; Transmit the address byte

EERF004
    btfss   SSPSTAT, BF
    bra     EERF004
    
    movf    SSPBUF, W           ; read byte
    movwf   scanLat

    movwf   SSPBUF              ; Transmit dummy byte to clock in data

EERF005
    btfss   SSPSTAT, BF
    bra     EERF005
    
    movf    SSPBUF, W           ; read byte
    movwf   scanLat+1

    movwf   SSPBUF              ; Transmit dummy byte to clock in data

EERF006
    btfss   SSPSTAT, BF
    bra     EERF006
    
    movf    SSPBUF, W           ; read byte
    movwf   scanLat+2

    movwf   SSPBUF              ; Transmit dummy byte to clock in data

EERF007
    btfss   SSPSTAT, BF
    bra     EERF007
    
    movf    SSPBUF, W           ; read byte
    movwf   scanLong

    movwf   SSPBUF              ; Transmit dummy byte to clock in data

EERF008
    btfss   SSPSTAT, BF
    bra     EERF008
    
    movf    SSPBUF, W           ; read byte
    movwf   scanLong+1

    movwf   SSPBUF              ; Transmit dummy byte to clock in data

EERF009
    btfss   SSPSTAT, BF
    bra     EERF009
    
    movf    SSPBUF, W           ; read byte
    movwf   scanLong+2

    bsf     LATC, 0
    bsf     LATC, 1
    return    
 


;*****************************************************************************        
;
;   Function :  EEPWrite64
;               Writes the 64 bytes, pointed to by FSR0, into EEPROM
;               using it's paged memory write feature.
;               The caller is responsible for ensuring that the 64 byte address
;               in eepAddH/L is correctly paged aligned.
;               This routine is responsible for managing the chip selects
;   Input:      EEPROM address in eepAddH + eepAddL. Data pointer in FSR0
;               
;
;   Output:     EEPROM updated
;
;*****************************************************************************        
EEPWrite64
    btfsc   eepAddH, 7
    bcf     LATC, 1
    
    btfss   eepAddH, 7
    bcf     LATC, 0
    
    nop
    nop
    
    movlw   EEPROM_WREN
    movwf   SSPBUF              ; Transmit the command byte

EEPW001
    btfss   SSPSTAT, BF
    bra     EEPW001
    
    movf    SSPBUF, W           ; Dummy read

    bsf     LATC, 0
    bsf     LATC, 1

    nop
    nop

    ; Enable appropriate chip select
    btfsc   eepAddH, 7
    bcf     LATC, 1
    
    btfss   eepAddH, 7
    bcf     LATC, 0
    
    nop
    nop


    movlw   EEPROM_WRITE
    movwf   SSPBUF              ; Transmit the command byte

EEPW0011
    btfss   SSPSTAT, BF
    bra     EEPW0011
    
    movf    SSPBUF, W           ; Dummy read


    movf    eepAddH, W
    movwf   SSPBUF              ; Transmit the address byte

EEPW002
    btfss   SSPSTAT, BF
    bra     EEPW002
    
    movf    SSPBUF, W           ; Dummy read

    movf    eepAddL, W
    movwf   SSPBUF              ; Transmit the address byte

EEPW003
    btfss   SSPSTAT, BF
    bra     EEPW003
    
    movf    SSPBUF, W           ; Dummy read
    
    movlw   D'64'
    movwf   TEMP        ; Use temp as the loop counter
    
EEPWLoop          
    movf    POSTINC0, W
    movwf   SSPBUF              ; Transmit the data byte

EEPW004
    btfss   SSPSTAT, BF
    bra     EEPW004
    
    movf    SSPBUF, W           ; Dummy read
    
    decfsz  TEMP, F
    bra     EEPWLoop
        
    bsf     LATC, 0
    bsf     LATC, 1

    ; Allow eeprom time to write the bytes
    movlw   0x01                
    call    UIWait10ms          

    return
    